#!/usr/bin/env python
#
# calv_memboot_smu_install.py
# This script is invoked on the memboot cards to install SMUs.
# Following tasks are done by this script:
#   1. Configure the right interface and check which RP is accessible
#   2. Copy over SWP file to /tmp
#   3. Read SWP and create a list of required PKGs
#   4. Copy over the PKGs from RP
#   5. Install RPMs (dependency check TBD)
#
#   Issues:
#   1. If the inst_mgr was in the middle of making changes to SWP when we 
#      copied the SWP, the SWP may not be up to date. In such case SWP match 
#      will fail and the card will be rebooted. The next reboot should work,
#      provided that the SWP is up to date on RP
#
# Copyright (c) 2014-2016, 2018 by Cisco Systems, Inc.
# All rights reserved.

import os
import sys
import tempfile
import shutil
import argparse
import time
import glob
import subprocess
from collections import OrderedDict
import re

def get_slot_id():
    """
        Get the slot ID to configute IP address
        This script is only going to run on XC/FC. 
        No need to handle LC which needs different address to be read
    """
    cmd = "pcimemread 0xe800000c 4| grep : | awk '{ print $3 }'"
    slot_str=os.popen(cmd).read()
    slot_str=slot_str.replace("000000","")
    
    slot = int (slot_str,16)
    slot %= 0x20
    if set_debug:
        print "My slot = %d"%slot
    return str(slot)

def config_interfaces(src_ip_list):
    """
        Configure eth0 for XC
        eth0/eth1 for FC
        This script is called from pxe_install_memboot.sh at this time
        the vritual bridge is not created and interface 'eobc' is not
        avaialble. So we are configuring eth0 (eth0 or eth1 is case of FC)
        with the geo-IP address 
    """
    global boardtype,set_debug,time_list
    ret_ip = ""
    ret_status = False
    ret_interface = ""
    src_ip = ""
   
    if set_debug:
        print "Executing ifconfig -a"
        sys.stdout.flush()
        os.system("ifconfig -a")

    slot = get_slot_id()
    ip = "127.1.2." + slot
    print "My IP : %s"%ip

    if (boardtype.lower() == 'xc'):
        interfaces = ["eth0"]
    else:
        # eth1 is connected to XC0, so try it first
        interfaces = ["eth1","eth0"]

    unconfig_all_interfaces(interfaces)

    for i in interfaces :
        print "Trying interface: %s"%i
        cmd = "ifconfig " + i + " " + ip + " netmask 255.255.0.0 up"
        ret = os.system(cmd)
        
        if (ret == 0) :
            (reachable,src_ip) = get_src_rp_ip(src_ip_list)
            if (reachable):
                ret_status = True    
                ret_interface = i
                break
            else:
                unconfig_interfaces(i)

    sys.stdout.flush()
    if (ret_status):
        print "Configured " + ret_interface + " with %s "%ip
    else :
        print "Could not configure interfaces !"

    if set_debug:
        print "Executing ifconfig -a"
        sys.stdout.flush()
        os.system("ifconfig -a")
    
    time_list.append(time.time())
    return (ret_status,ret_interface,src_ip)

def unconfig_interfaces(interface):
    """
        Down the interface
        This script is called from pxe_install_memboot.sh at this time
        the vritual bridge is not created and interface 'eobc' is not
        avaialble. So we are configuring eth0 (eth0 or eth1 is case of FC)
        with the geo-IP address. Need to flush the interface here once the
        required files have been copied.
    """
    if interface is "":
        return 

    print "Unconfiguring " + interface 
    cmd = "ifconfig " + interface + " down"
    sys.stdout.flush()
    os.system(cmd)

def unconfig_all_interfaces(interfaces):
    """
        Down all the interfaces 
    """
    for i in interfaces :
        unconfig_interfaces(i)

def check_status_of_httpd(ip):
    """
        Check the status of initrd file to determin if the httpd
        is running on the given RP
    """
    global is_host_smu,set_debug
    stat_file_name = "sysadmin-nbi-initrd.img"

    sys.stdout.flush()
    ret = os.system("wget -q --version > /dev/null 2>&1")
    if (ret != 0):
        # Busybox wget needs this option
        wget_stat_opt = " -s "
    else:
        # this is generic wget option to stat a file
        wget_stat_opt = " --spider "

    wget_stat_cmd = "wget -q " + wget_stat_opt +" http://"+ ip + "/" + stat_file_name
    sys.stdout.flush()
    if (set_debug):
        print "wget_stat_cmd = %s"%wget_stat_cmd
    ret = os.system(wget_stat_cmd)
    return (ret)

def get_src_rp_ip(src_ip_list):
    """
        This function pings the RPs to check which RP
        can be used to copy over SWP and SMU pkgs.
    """
    reachable=False
    src_ip=""
    # Try only one packet and timeout is 1 sec
    ping_cmd="ping -c 1 -W 1 "
    for ip in src_ip_list: 
        cmd = ping_cmd + ip
        sys.stdout.flush()
        ret = os.system(cmd)
        if (ret == 0):
            ret = check_status_of_httpd(ip)
            if (ret == 0):
                reachable=True
                src_ip=ip
                print "%s is reachable!"%(ip)
                break
            else:
                print "%s is reachable but wget did not get response !"%(ip)
        else:
            print "%s is not reachable (cmd = %s)!"%(ip,cmd)
    
    if (not reachable):
        print "None of the RPs are reachable !"
    
    return reachable,ip    

def copy_swp_from_rp(user,src_ip,swp_src_full_path,swp_dest_full_path):
    """
        This function copies over the SWP from the RP
    """
    global time_list
    if (os.path.exists(swp_dest_full_path)):
        os.remove(swp_dest_full_path)

    print "Initiating wget for SWP"
    #cmd = "wget http://" + src_ip + "/" + swp_src_full_path + " -O " + swp_dest_full_path
    cmd = "wget -q http://" + src_ip + "/" + swp_src_full_path + " -P " + os.path.dirname(swp_dest_full_path)
    
    print "cmd = %s "%cmd
    
    sys.stdout.flush()
    ret = os.system(cmd)
  
    if ret == 0:
        print "Received SWP successfully !"
    else:
        print "Error receiving SWP !"

    time_list.append(time.time())
    return (ret == 0)

def is_cisco_pkg(pkg_name):
    """
    Returns True if the given pkg is a Cisco Pkg
    """
    global platform_name, set_debug
    ret = False
    if (platform_name in pkg_name):
        ret = True
   
    if (set_debug):
        print "is_cisco_pkg(%s) = %s"%(pkg_name,ret)
    return ret    

def is_cisco_hostos_pkg(pkg_name):
    """
        Check if the package is cisco hostos package
    """
    ret = False
    if (is_cisco_pkg(pkg_name) and ("-hostos-" in pkg_name)) :
        ret = True
    if (set_debug):
        print "is_cisco_hostos_pkg(%s) = %s "%(pkg_name,ret) 
    return ret    

def extract_tp_package_name(pkg_name):
    """
        Exteact the package name from the <pkg>-<version>-<release>
    """
    global set_debug
    name = pkg_name.rsplit('-',2)[0]
    return name

def is_tp_smu_applicable_on_arch(pkg_name):
    """
        Returns True is the given TP pkg is valid on 'this' card
    """
    global set_debug, tp_host_rpm_list
    ret = False

    for pkg_name_arch in tp_host_rpm_list:
        name = pkg_name_arch.split(':')[0]
        if (pkg_name == name):
            ret = True
            break

    if (set_debug):
        if (ret):
            print "%s is applicable on this card"%pkg_name
        else:
            print "%s is NOT applicable on this card"%pkg_name

    return ret

def tp_smu_get_arch(pkg_name):
    """
        Returns the 'arch' of the given TP pkg
    """
    global set_debug, tp_host_rpm_list
    ret_arch = ''

    for pkg_name_arch in tp_host_rpm_list:
        name = pkg_name_arch.split(':')[0]
        arch = pkg_name_arch.split(':')[1]
        
        if (pkg_name == name):
            ret_arch = arch
            break

    if (set_debug):
        print "Architecture of %s is %s"%(pkg_name,ret_arch)

    return ret_arch

def get_pkg_arch(pkg_name):
    """
        Returns the 'arch' of the given pkg
    """
    arch = get_local_arch()
    if (not is_cisco_pkg(pkg_name)):
        arch = tp_smu_get_arch(extract_tp_package_name(pkg_name))

    return arch    

def is_host_vm_pkg(pkg_name):
    """
        Check if the given package is host VM package and valid for the architecture
    """
    global set_debug
    
    ret = False

    if (is_cisco_hostos_pkg(pkg_name) and ((".host" in pkg_name))):
        ret = True
    elif (not is_cisco_pkg(pkg_name) and (".host" in pkg_name)):
        stripped_name = extract_tp_package_name(pkg_name)
        if (set_debug):
            print "Stripped name = %s"%stripped_name

        if (is_tp_smu_applicable_on_arch(stripped_name)) :
            ret = True
    
    if (ret):
        print "%s is host VM package"%pkg_name
    else:
        print "%s is not host VM package"%pkg_name

    return ret

def is_calvados_vm_pkg(pkg_name):
    """
        Check if the given package is for Calvados VM
    """
    global set_debug
    ret = False

    if (is_cisco_hostos_pkg(pkg_name) and ((".admin" in pkg_name))):
        if (set_debug):
            print "%s is Calvados VM package"%pkg_name
        ret = True
    elif (not is_cisco_hostos_pkg(pkg_name) and is_cisco_pkg(pkg_name)):
        if (set_debug):
            print "%s is Calvados VM package"%pkg_name
        ret = True
    elif (not is_cisco_pkg(pkg_name) and (".admin" in pkg_name)):
        stripped_name = extract_tp_package_name(pkg_name)
        if (set_debug):
            print "Stripped name = %s"%stripped_name

        if (is_tp_smu_applicable_on_arch(stripped_name)) :
            if (set_debug):
                print "%s is host VM package"%pkg_name
            ret = True
    
    return ret

def get_local_arch():
    """
        Returns the local architecture of the card
        Right now we are hardcoding it to arm
    """
    return "arm"

def is_pkg_needed(pkg_name):
    """
        Check is the pkg needs to be downloaded
    """
    global is_host_smu, smu_list_host_fname, smu_list_calv_fname
    print "pkg_name = %s"%pkg_name    
    ret = False
    if is_host_smu :
        if (is_host_vm_pkg(pkg_name)):
            print "%s is needed"%pkg_name
            ret = True
            with open(smu_list_host_fname, "a") as f:
                f.write(pkg_name+"."+get_pkg_arch(pkg_name)+"\n")
    else:
        if (is_calvados_vm_pkg(pkg_name)):
            print "%s is needed"%pkg_name
            ret = True 
            with open(smu_list_calv_fname, "a") as f:
                f.write(pkg_name+"."+get_pkg_arch(pkg_name)+"\n")
    return ret

def parse_swp_line_for_pkg_name(line):
    """
        Use this if TP packages and Cisco packages are listed
        in a flat list format
    """
    swp_pkg = ""
    pattern = re.compile(".(---- \d+:)")
    if (pattern.match(line)):
        swp_pkg = line.split(': ')[1]
        swp_pkg = swp_pkg.split()[0]
        swp_pkg = swp_pkg.replace(' ','')
        swp_pkg = swp_pkg.replace('\n','')
    return swp_pkg

def parse_swp_line_for_sp_name(line):
    """
        Use this for checking if SP is present in SWP 
    """
    swp_sp = ""
    pattern = re.compile(".(---- SP \d+:)")
    if (pattern.match(line)):
        swp_sp = line.split(': ')[1]
        swp_sp = swp_sp.split()[0]
        swp_sp = swp_sp.replace(' ','')
        swp_sp = swp_sp.replace('\n','')
    return swp_sp

def copy_sp_info_file_to_root(src_path, sp_list):
    """
        Thsi function will copy sp_info files from src_path to / for SWP creation
    """
    global rootfs_path

    i = 0
    while (i < len(sp_list)):
        sp = sp_list[i]
        cmd = "cp " + src_path + "/" + sp + " " + rootfs_path
        print cmd
        sys.stdout.flush()
        if (os.system(cmd)):
            print "Failure in - %s"%cmd
        i += 1
    time_list.append(time.time())
    return 0

def get_active_pkg_list(swp_fname):
    """
        This function parses the SWP to generate list of required SMUs
    """
    global time_list, smu_list_host_fname, smu_list_calv_fname
    print "Parsing the SWP file"
    found_pkg=False
    pkg_list=[]
    is_valid_swp_file = False
    found_sp=False
    sp_list=[] 

    #Need to create following empty files, these files will contain the SMU 
    #list for the respective VM
    if is_host_smu :
        open(smu_list_host_fname, 'a').close()
    else:
        open(smu_list_calv_fname, 'a').close()
        
    try:
        with open(swp_fname, "r") as f_handle:
            for line in f_handle:
                #Check if the file is valid (Enhancement: Check MD5 )
                if (not is_valid_swp_file):
                    if ("Software Profile" in line):
                        is_valid_swp_file = True
                    continue    
                if found_pkg:
                    swp_pkg = parse_swp_line_for_pkg_name(line)
                    
                    if ((swp_pkg != "") and is_pkg_needed(swp_pkg)):
                        pkg_list.append(swp_pkg)
                            
                if ((found_pkg == False) and ("Packages" in line)):
                    found_pkg = True

                if ((found_sp == False) and ("Packages" in line)):
                    found_sp = True

                if found_sp:
                    sp_name = parse_swp_line_for_sp_name(line)
                    if (sp_name != ""):
                        sp_list.append(sp_name)

    except EnvironmentError:
        print "Error in reading file %s, file may not be available !"%swp_fname
    
    if (not is_valid_swp_file):
        print "The SWP file is malformed !"
    else :
        if (len(pkg_list) == 0):
            print "No packages need to be installed !"
        else:
            print "\n\nActive package count = %s"%len(pkg_list)
            print "The active package list : %s"%pkg_list
    
    time_list.append(time.time())
    return (is_valid_swp_file,pkg_list,sp_list)

def copy_pkgs_from_rp(user,src_ip,pkg_list,sp_list,src_path,dest_path,max_cli_len):
    """
        This fucntion copies over the required SMU pkgs from the RP
    """
    global time_list
    
    ret = 0 
    cur_len=0
    pkg_line=""
    pkg_line_list=[]

    i=0
    while (i < len(pkg_list)):
        pkg = pkg_list[i]
        
        arch = get_pkg_arch(pkg)

        if (cur_len+len(pkg)+len(arch)+1 < max_cli_len):
            if (cur_len > 0):
                pkg_line += ","
            pkg_line += pkg + "." + arch
            cur_len = len(pkg_line)
            i += 1
        else:
            pkg_line_list.append(pkg_line)
            cur_len=0
            pkg_line=""

    ''' service pack list handling '''
    i = 0
    while (i < len(sp_list)):
        sp = sp_list[i]

        if (cur_len+len(sp)+1 < max_cli_len):
            if (cur_len > 0):
                pkg_line += ","
            pkg_line += sp
            cur_len = len(pkg_line)
            i += 1
        else:
            pkg_line_list.append(pkg_line)
            cur_len=0
            pkg_line=""

    pkg_line_list.append(pkg_line)
    
    print "Starting Wget for :",pkg_line_list       

    wget_cmd_base = "wget -q http://" + src_ip + "/install_repo/gl/calvados/" 
    for pkgs in pkg_line_list:
        if ',' in pkgs:
            wget_cmd = wget_cmd_base + "{" + pkgs + "} -P " + dest_path + "/" 
        else:
            wget_cmd = wget_cmd_base + pkgs + " -P " + dest_path + "/" 
        print wget_cmd
        sys.stdout.flush()
        ret = os.system(wget_cmd)
        if (ret):
            print "Error in Wget:",wget_cmd
            print "Stopping Wget !"
            break

    copy_sp_info_file_to_root(dest_path, sp_list) 

    time_list.append(time.time())
    return (ret == 0)

def get_rpm_exclude_path(pkg_path):
    """
        Calculate the exclude path
    """
    global rootfs_path,boardtype

    cmd_supcard = "rpm -qp --qf '%{SUPPCARDS}' "+ pkg_path
    cmd_prefix  = "rpm -qp --qf '%{PREFIXES}' " + pkg_path
    cmd_name    = "rpm -qp --qf '%{NAME}' " + pkg_path
    cmd_ver     = "rpm -qp --qf '%{VERSION}' " + pkg_path
    cmd_release = "rpm -qp --qf '%{RELEASE}' " + pkg_path

    suppcards=os.popen(cmd_supcard).read().split(',')
    prefix=os.popen(cmd_prefix).read()
    pkg_name=os.popen(cmd_name).read()
    version=os.popen(cmd_ver).read()
    release=os.popen(cmd_release).read()
    
    excl_path=prefix + "/" + pkg_name +  "-" + version + "-" + release

    #Generate a list of excluded cards
    excl_cards=[]
    for card in suppcards:
        if ((card.lower() != boardtype.lower()) and (card.lower() != "all")):
            excl_cards.append(card)
  
    #Generate a list of excludepaths
    excl=[]
    for card in excl_cards:
        excl_str="--excludepath=" +excl_path + "/" + card
        excl.append(excl_str)
    
    exclude= " ".join(excl)
    #print "exclude = ",exclude
    return exclude


def replace_orig_hostos_pkg_rpm_with_smu(pkg_path,pkg,arch):
    """
        Need to replace original hostos_pkg.host rpm with the SMU of that RPM
        This will be installed later in memboot_install.sh
    """
    global rootfs_path
    print "This is hosos_pkg SMU rpm for host, moving it to /rpm/. It will be installed later"
    cmd = "rm " + rootfs_path + "/rpm/*hostos*.host.*.rpm " 
    cmd += "; cp " + pkg_path + "/" + pkg + "." + arch + " "  + \
            rootfs_path + "/rpm/" + pkg + "." + arch + ".rpm"
                
    if (os.system(cmd)):
        print "Failure in - %s"%cmd
    return 0    


def sort_rpms(pkg_list):
    '''
        Separate TP and Cisco RPMs
    '''
    cisco_pkg_list = []
    tp_pkg_list = []

    for pkg in pkg_list:
        if (is_cisco_pkg(pkg)):
            cisco_pkg_list.append(pkg)
        else:
            tp_pkg_list.append(pkg)

    return tp_pkg_list,cisco_pkg_list


def install_tp_rpms(pkg_dest_path,tp_pkg_list):
    '''
        Install all the TP SMU rpms in one command.
        commandline length is 128Kbytes, so even if we consider 128 bytes per 
        package name, we can install aroung 1024 packages at once.
            xargs --show-limits :
                Your environment variables take up 345 bytes
                POSIX upper limit on argument length (this system): 2094759
                POSIX smallest allowable upper limit on argument length (all systems): 4096
                Maximum length of command we could actually use: 2094414
                Size of command buffer we are actually using: 131072
    '''
    global rootfs_path,boardtype,time_list,is_chroot, is_host_smu,chroot_bin_path 
    inst_rpm_inst_failed_count = 0
    
    print "\nInitiating TP RPM installs :"
    tp_pkg_list = [pkg + "." + get_pkg_arch(pkg) for pkg in tp_pkg_list]
    rpm_opt = " -Uv --replacepkgs --nodeps --force "

    if (is_chroot is False) :
        os.chdir(pkg_dest_path)
        tp_pkg_list_str = (" ".join(tp_pkg_list))
        cmd = "time rpm --root " + rootfs_path + rpm_opt +" " + tp_pkg_list_str
    
    else:
        pkg_jail_path = pkg_dest_path.replace(rootfs_path,"")
        chroot_bin_jail_path = chroot_bin_path
        if (is_host_smu) :
            chroot_bin_jail_path = rootfs_path + chroot_bin_path
       
        tp_pkg_list = [pkg_jail_path + "/" +pkg for pkg in tp_pkg_list]
        tp_pkg_list_str = (" ".join(tp_pkg_list))
        cmd = "time " + "D=/ " + chroot_bin_jail_path + " " + rootfs_path + " rpm " + rpm_opt + \
              tp_pkg_list_str
    
    print cmd
    sys.stdout.flush()
    rpm_ret = os.system(cmd)
    if (rpm_ret != 0):
        inst_rpm_inst_failed_count = len(tp_pkg_list) 
        print "!!!!!!!!!!!!!!!!!!! Could not install TP RPMS !!!!!!!!!!!!!!!!!!!!\n"
        print tp_pkg_list
    else:
        print "TP RPMs installed successfully :"
        print tp_pkg_list

    return inst_rpm_inst_failed_count 

def install_cisco_rpms(pkg_dest_path,cisco_pkg_list):
    '''
        Install Cisco SMUs one by one
    '''
    global rootfs_path,boardtype,time_list,is_chroot, is_host_smu,chroot_bin_path, lazy_host_install
    inst_rpm_inst_failed_count = 0
    print "\nInitiating Cisco RPM installs :"
    for pkg in cisco_pkg_list:
        print "Installing %s"%pkg
        
        arch = get_pkg_arch(pkg)
        rpm_opt = ""
        exclude = get_rpm_exclude_path(pkg_dest_path+"/" + pkg + "." + arch)
        
        if (not is_cisco_hostos_pkg(pkg)):
            rpm_opt = " -i --nodeps --force "
        else:
            rpm_opt = " -Uv --replacepkgs --nodeps --force "

        if (is_chroot is False) :
            cmd = "time rpm --root " + rootfs_path + rpm_opt +exclude + " " + \
                  pkg_dest_path + "/" + pkg + "." + arch 
            if (is_cisco_hostos_pkg(pkg) and is_host_smu and lazy_host_install):
                replace_orig_hostos_pkg_rpm_with_smu(pkg_dest_path,pkg,arch)
                continue
        else:
            pkg_jail_path = pkg_dest_path.replace(rootfs_path,"")
            chroot_bin_jail_path = chroot_bin_path
            if (is_host_smu) :
                chroot_bin_jail_path = rootfs_path + chroot_bin_path

            if (is_cisco_hostos_pkg(pkg) and is_host_smu and lazy_host_install):
                replace_orig_hostos_pkg_rpm_with_smu(pkg_jail_path,pkg,arch)
                continue
            else:    
                cmd = "time D=/ " + chroot_bin_jail_path + " " + rootfs_path +" rpm " + rpm_opt + exclude + " " +\
                      pkg_jail_path + "/" + pkg + "." + arch 
            
        print cmd
        sys.stdout.flush()
        rpm_ret = os.system(cmd)
        if (rpm_ret != 0):
            inst_rpm_inst_failed_count += 1 
            print "!!!!!!!!!!!!!!!!!!! Could not install %s !!!!!!!!!!!!!!!!!!!!"%pkg
        else:
           print "%s installed successfully !"%pkg
        
    return inst_rpm_inst_failed_count

def install_rpms(pkg_dest_path,pkg_list):
    """
        Install the rpms
    """
    global time_list,rootfs_path 
    (tp_pkg_list,cisco_pkg_list) = sort_rpms(pkg_list)
    
    inst_rpm_inst_failed_count = 0
    
    cmd = "mount --rbind /proc " + rootfs_path + "/proc" 
    sys.stdout.flush()
    print "################## Bind mounting /proc ######: %s"%cmd
    rpm_ret = os.system(cmd)
    if (rpm_ret != 0):
        print "bind mount failed for /proc"
    cmd = "mount --rbind /sys " + rootfs_path + "/sys" 
    sys.stdout.flush()
    print "################## Bind mounting /sys ######: %s"%cmd
    rpm_ret = os.system(cmd)
    if (rpm_ret != 0):
        print "bind mount failed for /sys"

    if len(tp_pkg_list) > 0 :
        inst_rpm_inst_failed_count += install_tp_rpms(pkg_dest_path,tp_pkg_list)
    
    if len(cisco_pkg_list) > 0 :
        inst_rpm_inst_failed_count += install_cisco_rpms(pkg_dest_path,cisco_pkg_list)

    cmd = "umount -R -l " + rootfs_path + "/proc"
    cmd = "umount -R -l " + rootfs_path + "/sys"
   
    time_list.append(time.time())
    return inst_rpm_inst_failed_count    

def clean_rpm_db():
    """
        Rebuild the RPM DB
    """
    global rootfs_path, is_clean_rpm_db, time_list, set_debug
    
    if (is_clean_rpm_db):
        print "Cleaning RPM DB..."
        for db_name in glob.glob(rootfs_path + '/rpmdb/lib/rpm/__db*.*'):
            if os.path.isfile(db_name):
                os.remove(db_name)

        for db_name in glob.glob(rootfs_path + '/var/lib/rpm/__db*.*'):
            if os.path.isfile(db_name):
                os.remove(db_name)
    else:
        if (set_debug):
            print "is_clean_rpm_db = False, not cleaning the RPM DB !"
    
    time_list.append(time.time())

def rebuild_rpm_db():
    """
        Rebuild RPM DB
    """
    global rootfs_path, is_rebuild_rpm_db, time_list, set_debug
    
    if (is_rebuild_rpm_db):
        cmd = "rpm --root " + rootfs_path +" --rebuilddb"
        print "cmd = %s \n"%cmd
        sys.stdout.flush()
        os.system(cmd)
    else:
        if (set_debug):
            print "is_rebuild_rpm_db = False, not rebuilding the RPM DB !"
    
    time_list.append(time.time())

def print_time_analysis(op_enable_dict):
    """
        Print time analysis
    """
    global time_list
    print "\nTime analysis : \n"

    i = 0
    for key, value in op_enable_dict.iteritems():
        if (i >= len(time_list)):
            break
        if i == 0:
            i = i + 1
            continue
        if (value):
            print " %s took %s Seconds "%(key,str(time_list[i]-time_list[i-1]))
        i = i + 1

    print "\n Total time = %s Seconds \n"%(str(time_list[len(time_list)-1] - time_list[0]))

def process_smu_install():
    """
        All the tasks are done in this method:
           1. Configuring the interfaces
           2. Figuring out which RP to use
           3. Get the SWP
           4. Parse the SWP
           5. Fetch the SMU files
           6  SMU Installation
    """
    global rootfs_path,boardtype,set_debug,time_list,smu_download_path
    
    #Length of the file list which can be use in one Wget session
    #saving 124 bytes for 'wget -q http://127....' etc.
    max_cli_len=900 

    RP0_IP_ADDR="127.1.1.27"
    RP1_IP_ADDR="127.1.1.28"
    src_ip_list=[RP0_IP_ADDR,RP1_IP_ADDR]
    
    swp_fname="clos-master-swprofile-active.txt"
    swp_src_full_path="/install_repo/gl/instdb/" + swp_fname
    swp_dest_path="/tmp/"
    swp_dest_full_path=swp_dest_path + swp_fname
    

    #Install repository path on the RP. Make sure that repo is mounted on the host of RP
    pkg_src_path="/install_repo/gl/calvados/"
  
    if (is_chroot) :
        pkg_dest_path = rootfs_path + "/tmp_smu/"  
        if not os.path.exists(pkg_dest_path):
            os.makedirs(pkg_dest_path)
    elif (smu_download_path is not None):
        pkg_dest_path = smu_download_path
        if not os.path.exists(pkg_dest_path):
            os.makedirs(pkg_dest_path)
    else :
        #Temp dir for copied over rpms
        pkg_dest_path=tempfile.mkdtemp()

    pkg_dest_path = os.path.abspath(pkg_dest_path) 
    print "SMU download path : %s"%pkg_dest_path

    user="root"
    ret = False
    interface = ""
    src_ip = ""
    pkg_list = []
    status = False
    sp_list = []
    
    #Operations lookup
    op_enable_dict = OrderedDict()
    op_enable_dict['Start']                = True
    op_enable_dict['config_interfaces()']  = True
    op_enable_dict['copy_swp_from_rp()']   = True
    op_enable_dict['get_active_pkg_list()']= True
    op_enable_dict['copy_pkgs_from_rp()']  = True
    op_enable_dict['clean_rpm_db()']       = is_clean_rpm_db
    op_enable_dict['rebuild_rpm_db()']     = is_rebuild_rpm_db 
    op_enable_dict['install_rpms()']       = True

    time_list[:]=[]
    time_list.append(time.time())
    try:
        # Operation 1 - config_interfaces()
        (status,interface,src_ip) = config_interfaces(src_ip_list)
        if (status != True):
            ret = False
            raise Exception('config_interfaces()')
        
        # Operation 2 - copy_swp_from_rp()
        ret = copy_swp_from_rp(user,src_ip,swp_src_full_path,swp_dest_full_path)
        if (not ret):
            raise Exception('copy_swp_from_rp()')
        
        # Operation 3 - get_active_pkg_list()
        (status,pkg_list,sp_list) = get_active_pkg_list(swp_dest_full_path)
        if (status == False):
            ret = False
            raise Exception('get_active_pkg_list()')
        elif ((len(pkg_list) == 0) and (len(sp_list) == 0)):
            ret = True
            raise Exception('get_active_pkg_list()')
        
        # Operation 4 - copy_pkgs_from_rp()
        ret = copy_pkgs_from_rp(user,src_ip,pkg_list,sp_list,pkg_src_path,pkg_dest_path,max_cli_len)
        if (not ret):
            raise Exception('copy_pkgs_from_rp()')
        
        # Operation 5 - clean_rpm_db()
        clean_rpm_db()

        # Operation 6 - rebuild_rpm_db()-- this is right now disabled by is_rebuild_rpm_db=False
        rebuild_rpm_db()
        
        # Operation 7 - install_rpms()
        inst_rpm_inst_failed_count = install_rpms(pkg_dest_path,pkg_list)
        if (inst_rpm_inst_failed_count == 0):
            #RPMs have been installed
            ret = True
        else:
            ret = False
            raise Exception('install_rpms()')

    except Exception, e:
        if (ret  == False): 
            print "SMUs could not be installed , Exception in ",e
    else:
        print "SMUs installation is completed !"
    finally:
        unconfig_interfaces(interface)
        shutil.rmtree(pkg_dest_path)

    print_time_analysis(op_enable_dict)

    if set_debug:
        print "process_smu_install() is returning %d \n"%ret
    
    return ret

def get_tp_host_rpm_name_list():
    """
       Generate the list of TP pkgs installed on the card 
    """
    global platform_name, is_chroot, rootfs_path, chroot_bin_path

    if (is_chroot is False) :
        p = subprocess.Popen(['rpm', '-qa', '--qf', '%{NAME}:%{ARCH}\n'],
                             stdout=subprocess.PIPE, 
                             stderr=subprocess.PIPE)
    else:
        chroot_bin_jail_path = chroot_bin_path
        if (is_host_smu) :
            chroot_bin_jail_path = rootfs_path + chroot_bin_path

        if set_debug:
            print "chroot_bin_jail_path = %s"%chroot_bin_jail_path

        p = subprocess.Popen([chroot_bin_jail_path,rootfs_path,'rpm', '-qa', '--qf', '%{NAME}:%{ARCH}\n'],
                             stdout=subprocess.PIPE, 
                             stderr=subprocess.PIPE)
    
    out, err = p.communicate()
    rpm_list = out.split("\n")
    rpm_list.remove("")

    remove_list = []
    for pkg in rpm_list:
        if (platform_name in pkg):
            remove_list.append(pkg)
    for pkg in remove_list:
        rpm_list.remove(pkg)

    return rpm_list

if __name__ == "__main__":
    
    #Global variables
    platform_name = "fretta"
    chroot_bin_path = "/usr/sbin/chroot"
    smu_list_host_fname = "/sysroot/tmp/smu_list_host.txt"
    smu_list_calv_fname = "/tmp/smu_list_calv.txt"
    
    set_debug           = False
    is_host_smu         = False
    is_chroot           = False
    is_clean_rpm_db     = True
    is_rebuild_rpm_db   = False

    #Commandline parser
    parser = argparse.ArgumentParser()
    parser.add_argument('--boardtype',        '-b',     required=True, help="Provide the boardtype")
    parser.add_argument('--rootfs_path',      '-c',     required=True, help="Provide the rootfs path")
    parser.add_argument('--smu_download_path','-d',     help="Provide the smu download path")
    parser.add_argument('--is_host_smu',      '-host',  action = 'store_true', help="Use this flag when dealing with the Host VM")
    parser.add_argument('--is_chroot',        '-chroot',action = 'store_true', help="Use this flag if chroot is needed for rpm installation")
    parser.add_argument('--verbose',          '-v',     action = 'store_true')
    parser.add_argument('--lazy_install',  '-nodel', action = 'store_false')
    args = parser.parse_args()

    boardtype=args.boardtype
    rootfs_path=args.rootfs_path
    smu_download_path = args.smu_download_path
    is_host_smu = args.is_host_smu
    is_chroot = args.is_chroot
    lazy_host_install = args.lazy_install
    if not lazy_host_install:
        smu_list_host_fname = "/tmp/smu_list_host.txt"
    set_debug = args.verbose
    platform_name = "ncs5500"

    print "\n--------------------------------------------------------------------------"
    print "                          Installing SMUs"
    print "--------------------------------------------------------------------------"

    print "Board Type = %s "%boardtype
    print "Rootfs Path = %s"%rootfs_path
    if (smu_download_path is not None):
        print "smu_download_path = %s"%smu_download_path

    print "is_host_smu = %s"%is_host_smu
    print "is_chroot = %s"%is_chroot
    print "lazy_host_install = %s"%lazy_host_install

    # For time profiling
    time_list=[]
    
    #Get the list of base rpms
    tp_host_rpm_list = get_tp_host_rpm_name_list()

    #Max number of tries in case of bad connection situations
    retry_count = 5

    ret = False
    for i in range (0,retry_count):
        print "\nTry count %d.........................................................\n"%(i+1)
        ret = process_smu_install()
        if (ret == True) :
            break
        time.sleep(5)

