#!/bin/bash
#
# Copyright (c) 2015, 2020 by cisco Systems, Inc.
# All rights reserved
# hushd_iofpga.sh - IOFPGA based register get/putter 
#

VERSION="0.0 2/25/2015" 

###############################################################################

# Pci address mapping

#

# figure out where the base pci address is for the iofpga 

# 

# Always done at the start of this module

#

IOFPGA_ID=$(lspci -nd 1137:0100 | tail -1 | awk '{ print $1 }');

if test -z $IOFPGA_ID; then

    IOFPGA_ID=$(lspci -nd 1137:0101 | tail -1 | awk '{ print $1 }');

    if test -z $IOFPGA_ID; then
        echo "Could not determine IOFPGA pci address vi lspci"

        exit 1
    fi
fi

IO_BASE=0x$(lspci -vs $IOFPGA_ID | egrep -w 'Memo|size=256M' |  awk '{ print $3 }');

IOFPGA_BASE=$(printf 0x%08X $(($IO_BASE + 0x08000000)) )



###############################################################################

# swap32

#

# 32 bit numeric swap

#

swap32() {

    if test $# != 1; then

        echo "syntax: $FUNCNAME value"

        exit 1

    fi

    

    value=$1                # input value

    

    v0=$(($value & 0xff))

    value=$((value >> 8))

    v1=$(($value & 0xff))

    value=$((value >> 8))

    v2=$(($value & 0xff))

    value=$(($value >> 8))

    v3=$(($value & 0xff))



    r=$(($v0 << 24 | $v1 << 16 | $v2 << 8 | $v3))

    echo $(printf 0x%08X $r)  # output in 0xhex

}



###############################################################################

# iofpga_read

#

# reads from pci, arg0 address

#

iofpga_read() {

    if test $# != 1; then

        echo "syntax: $FUNCNAME address" >> /tmp/hushd_platform_debug

        exit 1

    fi

        # pcimemread requires address in hex, ignores any leading 0x

    regaddr=$(printf 0x%08X $(($IOFPGA_BASE + $1)) )

    len=4

        # output of lspci is something like this  (want arg 12)

        # PCI memory 0xf0500000 align memory 0xf0500000 (mmap addr 0x7f3e230ad000(0x2000)) f0500000 : 11010000 ....

    v=$(pcimemread $regaddr $len )

    v1=$(echo $v | awk '{ print $12 }')   # can't combine above - multiline??

    v2=$((0x$v1))

    echo $(printf 0x%08x $v2) >> /tmp/hushd_platform_debug

}





###############################################################################

# iofpga_write

#

# reads from pci, arg0 address

#

iofpga_write() { 

    if test $# != 2; then

        echo "syntax: $FUNCNAME address value"

        exit 1

    fi

    regaddr=$(printf 0x%08X $(($IOFPGA_BASE + $1)) )

    value=$2

    len=4

    

      # echo regaddr=$regaddr len=$len value=$value

    pcimemwrite $regaddr $len $value > /dev/null      # get rid o output

    echo "read from $regaddr to confirm the write :" >> /tmp/hushd_platform_debug

    iofpga_read $1

    return

}





###############################################################################

# iofpga_dump

#

# reads from pci, arg1 address, arg2 length

#

iofpga_dump() { 

    if test $# != 2; then

        echo "syntax: $FUNCNAME address length"

        exit 1

    fi

        # pcimemread requires address in hex, ignores any leading 0x

    regaddr=$(printf 0x%08X $(($IOFPGA_BASE + $1)) )

    len=$(( $(($2 + 3)) & 0xfffc))  # len reasonable clipped and divisiable by 4

    len=$(printf 0x%08X $len)   # length also must be in hex

    pcimemread $regaddr $len >> /tmp/hushd_platform_debug

}






###############################################################################

# check SUP Arbitration Control Register --

# check bit 4 - active_sup is 1, active sup, Gamed mastership

# if bit 4 is set, then set bit 5 to 1 to relinquish the mastership


sup_arb_cntl() {


        # SUP Arbitration Control Register 

        # $IOFPGA_BASE + 0x00000330 SUP_ARB_CTRL_REG

        #    

        #             unused                           R/W, default 0

        #                   Offline                    R/W, 1=offline, 0=not offline

        #           /       /  active_sup              R, 1 = ctive sup, Gained mastership, 0 = not active sup
        #          /       /  / 

        #         /       /  /  which_active           R, which active[2:0] is a sample from the connector pin

        #       /       /  /   /    Im_active          R/W, 1 = start active supervisor arbitration through inter-locking logic, 0 = not in arbitration.

        #     /       /  /   /    /

        #  31-8      5  4  3-1  0


    regaddr=$(printf 0x%08X $(($IOFPGA_BASE + 0x00000330)) )

    len=4

    v=$(pcimemread $regaddr $len )

    v1=$(echo $v | awk '{ print $12 }')   # can't combine above - multiline??

    v2=$((0x$v1))

    v3=`echo $(printf 0x%08x $v2)`

    mask=0x10

    bit4=$((v2 & mask))

    echo "check bit 4 setting, $v3, bit4=$bit4" >> /tmp/hushd_platform_debug

    if [ ${bit4} != 0 ]; then

        bit5=0x20

        value=$((v2 | bit5))

        val2=`echo $(printf 0x%08x $value)`

        echo "set bit5 in register, val2 = $val2" >> /tmp/hushd_platform_debug

        pcimemwrite $regaddr $len $val2 > /dev/null      # get rid o output

        echo "read sup_arb control register back to confirm" >> /tmp/hushd_platform_debug

        iofpga_read 0x00000330

    fi

}






###############################################################################

# command_handler

#   

# command parser and handler

#

command_handler() {

    action=$1

    arg1=$2

    arg2=$3

    arg3=$4

    arg4=$5

    

    case $action in  

        mmap)   echo IOFPGA_ID=$IOFPGA_ID IOFPGA_BASE=$IOFPGA_BASE ;;

        d)      iofpga_dump $arg1 $arg2    ;;

        r)      iofpga_read $arg1          ;;

        w)      iofpga_write $arg1 $arg2   ;;

        or)     v=$(iofpga_read $arg1)

                v=$(($v | $arg2))

                iofpga_write $arg1 $v

                iofpga_read $arg1

                ;;

        and)    v=$(iofpga_read $arg1)

                v=$(($v & $arg2))

                iofpga_write $arg1 $v

                iofpga_read $arg1

                ;;

        wr)     iofpga_write $arg1 $arg2

                echo $(iofpga_read $arg1)

                ;;


    sup_arb_cntl)  # check SUP Arbitration Control Register -- 

                   # check bit 4 - active_sup is 1, active sup, Gamed mastership

                   # if bit 4 is set, then set bit 5 to 1 to relinquish the mastership
 
                   sup_arb_cntl 

                ;; 

   

    version)     echo $VERSION ;;                # show version

          *)
            echo "mmap                       shows pci mem mapping"

            echo "r        address           read address"

            echo "w        address value     new <-- value"

            echo "d        address len       dump regs at addr"

            echo "or       address value     new <-- old | value"

            echo "and      address value     new <-- old & value"

            echo "wr       address value     write/read value to address"

            echo "sup_arb_cntl               read SUP Arbitration Control Register, if bit 4 is set, then set bit 5 to relinquish the mastership"

            ;;

    esac

}



###############################################################################

# interactive loop

#   

interactive_loop() {

    while read -p "iofpga >" action arg1 arg2 arg3 arg4; do

        command_handler $action $arg1 $arg2 $arg3 $arg4

    done

}



###############################################################################

# main.... 

#

# start up loop if no arguments, otherwise do command

#

if test $# == 0; then

    interactive_loop

else

    command_handler $1 $2 $3 $4 $5

fi




