#!/bin/bash
#
# Copyright (c) 2015 by cisco Systems, Inc.
# All rights reserved
# hushd_platform.sh - wrapper over hushd_iofpga.sh
#

DBG_LOG=/tmp/hushd_platform_debug
IOFPGA_script=/opt/cisco/hostos/bin/hushd_iofpga.sh

# Load common init scripts that we can reuse routines to find out board type
# info
if [ -f /etc/init.d/mod_ins/module-load-functions ]; then
    . /etc/init.d/mod_ins/module-load-functions
fi

# Routine to find out the board type
function get_board_type() {
    ARCH=`arch`
    if [ "$ARCH" == "armv7l" ]; then
        BOARDTYPE=$(get_cctrl_board_type)
    else 
        BOARDTYPE="$(cat /proc/cmdline|awk -F"boardtype=" '{print $2}'|cut -d' ' -f1)"
        if [ -z ${BOARDTYPE} ];then
            BOARDTYPE="RP"
        fi
    fi
}

#
# Transmit POST code to SC via one-wire
#
# The POST code is set on bits 0:7 and bit 31 has to set to '1' for HW to
# transmit the post code to SC.
RP_IOFPGA_POST_CODE_REG_ADDR=0x250
LC_IOFPGA_POST_CODE_REG_ADDR=0x210
SC_IOFPGA_POST_CODE_REG_ADDR=0x30C
FC_IOFPGA_POST_CODE_REG_ADDR=0x214
START_TRANSMIT_BIT=0x80000000
function pd_send_post_code_to_sc () {
    local post_code=$(printf 0x%08X $(($START_TRANSMIT_BIT + $1)))
    local event_name=$2
    local reg_addr=0

    if [ "$BOARDTYPE" == "RP" ]; then
        reg_addr=$RP_IOFPGA_POST_CODE_REG_ADDR
    elif [ "$BOARDTYPE" == "LC" ]; then
        reg_addr=$LC_IOFPGA_POST_CODE_REG_ADDR
    elif [ "$BOARDTYPE" == "XC" ]; then
        reg_addr=$SC_IOFPGA_POST_CODE_REG_ADDR
    elif [ "$BOARDTYPE" == "FC" ]; then
        reg_addr=$FC_IOFPGA_POST_CODE_REG_ADDR
    else
       echo "Unkonwn board type of $BOARDTYPE" >> $DBG_LOG
    fi

    if [ "$reg_addr" != "0" ]; then
        echo "Sending $event_name event" >> $DBG_LOG
        $IOFPGA_script w $reg_addr $post_code
    fi
}

#
# Definition of HOST POST Codes
#
# Definition of all supported POST CODES is in the following header file:
# calvados/dc_common_pkg/drivers/card_mgr/ncs5k/include/cmgr_hal_ncs5k_post_code.h
POST_CODE_HOST_SHUTDOWN_STARTED=0x55
POST_CODE_HUSHD_SYSADMIN_VM_BOOT_TIMEOUT=0x90
POST_CODE_HUSHD_SYSADMIN_VM_HBLOSS=0x91

#
# Send POST_CODE_HUSHD_SYSADMIN_VM_HBLOSS code to SC
function notify_sysadmin_vm_hbloss_event () {
    pd_send_post_code_to_sc $POST_CODE_HUSHD_SYSADMIN_VM_HBLOSS "SysAdmin VM HB Loss"
}

#
# Send POST_CODE_HOST_SHUTDOWN_STARTED code to SC
function notify_host_going_down_event () {
    pd_send_post_code_to_sc $POST_CODE_HOST_SHUTDOWN_STARTED "Host Shutdown Started"
}

#
# Release HW arbitration mastership (only applicable for RP and SC)
function release_mastership () {

    if [ "$BOARDTYPE" == "RP" -o "$BOARDTYPE" == "XC" ]; then
        echo "$0: $IOFPGA_script sup_arb_cntl" >> $DBG_LOG
        $IOFPGA_script sup_arb_cntl
    fi
}


function print_usage() {
   echo "Usage"
   echo "hushd_platform.sh doorbell        : Notify admin vm failure to standby"
   echo "hushd_platform.sh failover        : Force RP failover through hardware"
   echo "hushd_platform.sh update_flash "
   echo "{primary|backup} boardtype        : Update boardtype in flash"
   echo "hushd_platform.sh get_bios_ver " 
   echo "boardtype flash_type              : Get BIOS version from flash"
}

###############################################################################
# command_handler
#   
# command parser and handler
#


command_handler() {
    action=$1
    arg1=$2
    arg2=$3

    echo "$0: action $action" >> $DBG_LOG

    case $action in  

        doorbell)
            # This is indication of SysAdmin VM heartbeat loss, so we need to
            # send out SYSADMIN_HBLOSS POST code and if we are running
            # from RP or SC, we have to release the mastership
            notify_sysadmin_vm_hbloss_event
            release_mastership
            ;;

        failover)
            # The "failover" term here is kind of misleading as this can
            # happen to non-RP cards and for those, there is no failover
            # concept.
            # This is basically called by hushd when it has decided not
            # to respawn SysAdmin VM, and it will reset the board after
            # collecting the VM core.
            # Here we just want to send a POST Code out about this event
            # and if we are running from RP or SC, we have to release the
            # mastership
            # NOTE: this option seems to be legacy hushd code that used
            #       to trigger expiring of CCC stage1 watchdog as way to
            #       send distress signal. With the addition of sysadmin
            #       VM down doorbell, this path should be really be
            #       deprecated. For now we will treat it like 'doorbell' case.
            notify_sysadmin_vm_hbloss_event
            release_mastership
            ;;

        update_flash)
            # Update of BIOS flash in Fretta can be done from Calvados, so
            # do not need to implement this.
            echo "update_flash function not support on Fretta" >> $DBG_LOG
            ;;

        get_bios_ver)
            # Getting BIOS version in Fretta can be done from Calvados, so
            # do not need to implement this.
            echo "get_bios_ver function not support on Fretta" >> $DBG_LOG
            ;;

        host_going_down)
            # Notify Master card-mgr that host is going down, so any reset
            # request should give host chance to go down gracefully before
            # doing the actual reset operation.
            notify_host_going_down_event

            # Mastership will be released from CCTRL KLM as when this is
            # callled the SystemAdmin VM is still actually running, the
            # Linux shutdown procedure will bring it down and then the
            # CCTRL KLM reboot hook will do the rest.
            ;;
        *)
            print_usage
            ;;
    esac
}


# First, set the BOARDTYPE variable
get_board_type

if [ ! -f $IOFPGA_script ]; then
    echo "ERROR: missing $IOFPGA_script" >> $DBG_LOG
    exit 1
fi

###############################################################################
# main.... 
#
# start up loop if no arguments, otherwise do command
#

if test $# == 0; then
    print_usage
else
    command_handler $1 $2 $3
fi

