#!/bin/bash
#
# May 2018, Abhishek Ramachandra
#
# Copyright (c) 2010-2018 by cisco Systems, Inc.
# All rights reserved.
#

source /etc/init.d/calvados_bootstrap.cfg

function set_dr_globals {
    readonly HOST_DISK1_LABEL=hostos_DISK_ONE
    readonly HOSTOS_DISK1_PATH=/misc/disk1
    readonly REC_IMG_PATH=/recovery
    readonly REC_IMG=system_image.iso
    readonly PART_1_NAME=partition01
    readonly PART_REC_NAME=partitionDR
    readonly DR_LOGFILE=/var/log/dr.log
}

# Verbose output should always go to the file (for debug-ability)
function dr_log_file {
    local log_msg=${1}
    local time_str=$(date)
    echo "${time_str}: ${log_msg}" >> $DR_LOGFILE
}
readonly -f dr_log_file

function get_dev_by_label {
    local label=${1}
    echo `blkid -L ${label}`
}

function check_if_partitioned {
    set_dr_globals
    local dr_disk=/dev/${MAIN_DISK[0]}
    local dr_dev=$(get_dev_by_label "${RECOVERY_PART_LABEL}")
    
    # Check if EFR partition with matching Recovery filesystem label exists
    if [ -z "${dr_dev}" ]; then
        dr_log_file "No filesystem with label $RECOVERY_PART_LABEL exists"
        echo 0 && return
    fi
    
    # Get partition label of DR EFI partition
    local part_label=`blkid ${dr_dev} -s PARTLABEL -o value`
    if [ -z "${part_label}" ]; then
        dr_log_file "Unable to get partition label for $dr_dev"
        echo 0 && return
    fi

    # Check if DR EFI partition has an entry in GPT
    local part_entry=$(parted -s ${dr_disk} print < /dev/null | grep "${part_label}")
    if [ -z "${part_entry}" ]; then
        dr_log_file "No entry for DR EFI partition in GPT"
        echo 0 && return
    fi
    
    # Check if DR EFI partition is of type vfat and is marked bootable
    read fstype boot_flag <<< $(echo $part_entry | awk '{print $5 " " $7}')
    if [ $fstype != "fat16" -o $boot_flag != "boot" ]; then
        dr_log_file "DR EFI partition is not of type vfat or not bootable"
        echo 0 && return
    fi
    
    dr_log_file "Disk $dr_disk is already partitioned for DR"
    echo 1 && return 1
}

function mount_debug_partition() {
    . /etc/init.d/mount_debug.sh
    set -e
    # The above script always mounts DP(debug partition) at /tmp/DEBUG
    # /tmp/DEBUG is link to the actual mount point which will
    # be like /tmp/bakelog.n29446
    echo "MNT_POINT=/tmp/DEBUG"
    set +e
}

function unmount_debug_partition() {
    set -e
    debugmnt=$(mount | grep "/tmp/bakelog" | grep "/dev/loop"|cut -d" " -f3)
    debugloop=$(mount | grep "/tmp/bakelog" | grep "/dev/loop"|cut -d" " -f1)
    if [ -n "${debugloop}" ]; then
       umount ${debugmnt}
       losetup -d ${debugloop}
       rm -rf ${debugmnt}
       rm /tmp/DEBUG
    fi
    set +e 
}

