#!/usr/bin/python

import os
import fcntl
import sys
from bcm_sdk import *

# Defines for interface creation
oper = sys.argv[1]
pci_bus = int(sys.argv[2],0)
ifname = sys.argv[3]

# Helper functions
def send_ioctl(msg):
    fd = os.open("/dev/linux-bcm-knet", os.O_RDWR)
    if fd == -1:
        raise Exception("linux-bcm-knet not available")
    rc = fcntl.ioctl(fd, 0, msg, True)
    if rc != 0:
        raise Exception("ioctl failed")
    if msg.len < sizeof(kcom_msg_hdr_t):
        raise Exception("ioctl results too short (len = %d)" % (msg.len))
    if msg.rc != 0:
        raise Exception("ioctl resulted in error (rc = %d)" % (msg.rc))
    if msg.msg.hdr.type != KCOM_MSG_TYPE_RSP:
        raise Exception("unexpected response from ioctl (hdr.type = %d)" % (msg.msg.hdr.type))
    if msg.msg.hdr.status != KCOM_E_NONE:
        raise Exception("unexpected status from ioctl (hdr.type = %d)" % (msg.msg.hdr.status))
    os.close(fd)

def bcm_knet_version():
    msg = bkn_ioctl_t(kcom_msg_version_t)
    send_ioctl(msg)
    return msg.msg.version, msg.msg.netif_max, msg.msg.filter_max

def bcm_knet_hw_reset(unit, channels=0):
    msg = bkn_ioctl_t(kcom_msg_hw_reset_t)
    msg.msg.hdr.unit = unit
    msg.msg.channels = channels
    send_ioctl(msg)

def bcm_knet_cos(unit, channel, value, mask=0xffffffffffffffff):
    msg = bkn_ioctl_t(kcom_msg_cos_t)
    msg.msg.hdr.unit = unit
    msg.msg.channel  = channel
    msg.msg.mask     = mask
    msg.msg.value    = value
    send_ioctl(msg)
    return msg.msg.value

def bcm_knet_hw_init(unit, cmic_type='m', dcb_type=0, dcb_size=64, cdma_channels=7):
    msg = bkn_ioctl_t(kcom_msg_hw_init_t)
    msg.msg.hdr.unit      = unit
    msg.msg.cmic_type     = ord(cmic_type)
    msg.msg.dcb_type      = dcb_type
    msg.msg.dcb_size      = dcb_size
    msg.msg.pkt_hdr_size  = 4
    msg.msg.dma_hi        = 0x10000000
    msg.msg.cdma_channels = cdma_channels
    send_ioctl(msg)

def bcm_knet_netif_list(unit):
    msg = bkn_ioctl_t(kcom_msg_netif_list_t)
    msg.msg.hdr.unit = unit
    send_ioctl(msg)
    return msg.msg.id[0:msg.msg.ifcnt]

def bcm_knet_netif_get(unit, id):
    msg = bkn_ioctl_t(kcom_msg_netif_get_t)
    msg.msg.hdr.unit = unit
    msg.msg.hdr.id   = id
    send_ioctl(msg)
    return { "name":  msg.msg.netif.name,
             "mac":   msg.msg.netif.macaddr,
             "vlan":  msg.msg.netif.vlan,
             "type":  msg.msg.netif.type,
             "id":    msg.msg.netif.id,
             "flags": msg.msg.netif.flags,
             "port":  msg.msg.netif.port,
           }

def bcm_knet_get_unit(pci_bus):
    msg = bkn_ioctl_t(kcom_msg_get_unit_t)
    msg.msg.pdev_bus = pci_bus
    msg.msg.pdev_dev = 0xFF
    msg.msg.pdev_func = 0xFF
    send_ioctl(msg)
    return msg.msg.hdr.unit

def bcm_knet_netif_create(unit, name, type, vlan, flags=0, port=-1, qnum=0, macaddr=None):
    msg = bkn_ioctl_t(kcom_msg_netif_create_t)
    msg.msg.hdr.unit = unit
    msg.msg.netif.type = type
    msg.msg.netif.name = name
    msg.msg.netif.vlan = vlan
    msg.msg.netif.flags = flags
    if macaddr:
        for i in range(6):
            msg.msg.netif.macaddr[i] = macaddr[i]
    if type == KCOM_NETIF_T_PORT:
        msg.msg.netif.port = port
        msg.msg.netif.qnum = qnum
    send_ioctl(msg)
    return { "id":   msg.msg.netif.id,
             "mac":  msg.msg.netif.macaddr,
             "name": msg.msg.netif.name }

def bcm_knet_filter_create(unit, desc=None, flags=0, priority=255,
                           dest=None, mirror=None,
                           oob=None, pkt=None):
    msg = bkn_ioctl_t(kcom_msg_filter_create_t)
    msg.msg.hdr.unit        = unit
    msg.msg.filter.type     = KCOM_FILTER_T_RX_PKT
    msg.msg.filter.priority = priority
    if desc:
        msg.msg.filter.desc     = desc
    msg.msg.filter.flags    = flags
    if dest:
        msg.msg.filter.dest_type  = dest[0]
        msg.msg.filter.dest_id    = dest[1]
        msg.msg.filter.dest_proto = dest[2]
    if mirror:
        msg.msg.filter.mirror_type  = mirror[0]
        msg.msg.filter.mirror_id    = mirror[1]
        msg.msg.filter.mirror_proto = mirror[2]
    filter_size=0
    if oob:
        msg.msg.filter.oob_data_offset = oob[0]
        for i in range(len(oob[1])):
            msg.msg.data.b[filter_size + i] = oob[1][i]
        if len(oob) < 3:
            for i in oob[1]:
                msg.msg.mask.b[filter_size + i] = 0xff
        elif len(oob[1]) != len(oob[2]):
            raise Exception("oob filter mask(%d) and data(%d) not the same length" % (len(oob[2]), len(oob[1])))
        else:
            for i in range(len(oob[2])):
                msg.msg.mask.b[filter_size + i] = oob[2][i]
        filter_size += len(oob[1])
    if pkt:
        msg.msg.filter.pkt_data_offset = pkt[0]
        for i in range(len(pkt[1])):
            msg.msg.data.b[filter_size + i] = pkt[1][i]
        if len(pkt) < 3:
            for i in pkt[1]:
                msg.msg.mask.b[filter_size + i] = 0xff
        elif len(pkt[1]) != len(pkt[2]):
            raise Exception("pkt filter mask(%d) and data(%d) not the same length" % (len(pkt[2]), len(pkt[1])))
        else:
            for i in range(len(pkt[2])):
                msg.msg.mask.b[filter_size + i] = pkt[2][i]
        filter_size += len(pkt[1])
    send_ioctl(msg)
    return msg.msg.filter.id

def bcm_knet_netif_destroy(unit, netif_id):
    msg = bkn_ioctl_t(kcom_msg_netif_destroy_t)
    msg.msg.hdr.unit = unit
    msg.msg.hdr.id = netif_id
    send_ioctl(msg)

# Initialization/reinitialization sequence
unit = bcm_knet_get_unit(pci_bus)

if oper == "stop":
    id = bcm_knet_netif_list(unit)
    found = 0
    for i in id:
        netif = bcm_knet_netif_get(unit, i)
        if netif["name"] == ifname:
            netif_id = netif["id"]
            found = 1
            break
    if found:
        bcm_knet_netif_destroy(unit, netif_id)
    else:
        print "%s not found" % ifname
    exit(0)

if oper != "start":
   exit(0)

bcm_knet_version()
bcm_knet_hw_reset(unit)

sysfile = "/sys/bus/pci/devices/0000:%0.2x:00.0/device" % ( pci_bus )
try:
    with open(sysfile, 'r') as f:
        dev_family = int(f.readline(), 16) & 0xfff0
except:
    dev_family = 0x8670

if dev_family == 0x8690 or dev_family == 0x8800:
    print "KNET CMIC-X initialization for Jericho2 family"
    #cmic_type is uint, however, bcm-knet code uses it mainly to compare with 'x'/'m'
    bcm_knet_hw_init(unit, cmic_type='x', dcb_type=36, dcb_size=16, cdma_channels=0xff)
else:
    print "KNET CMIC-M initialization for Jericho family"
    #cmic_type is uint, however, bcm-knet code uses it mainly to compare with 'x'/'m'
    bcm_knet_hw_init(unit, cmic_type='m')
bcm_knet_cos(unit, 2, 0)
bcm_knet_cos(unit, 1, 0xffffff)
id = bcm_knet_netif_list(unit)
print id
for i in range(len(id)):
    print "netif %2d: id %2d" % (i, id[i])
found = 0
for i in id:
    netif = bcm_knet_netif_get(unit, i)
    print "%s: %02x%02x:%02x%02x:%02x%02x" % (netif["name"], netif["mac"][0], netif["mac"][1], netif["mac"][2], netif["mac"][3], netif["mac"][4], netif["mac"][5])
    if netif["name"] == ifname:
        netif_id = netif["id"]
        found = 1
        break
if not found:
    netif_id = bcm_knet_netif_create(unit, ifname, KCOM_NETIF_T_VLAN, 0)
    print "netif: id %d name %s mac: %02x%02x.%02x%02x.%02x%02x" % ( netif_id["id"], netif_id["name"], netif_id["mac"][0], netif_id["mac"][1], netif_id["mac"][2], netif_id["mac"][3], netif_id["mac"][4], netif_id["mac"][5])
    netif_id = netif_id["id"]
    filter_id = bcm_knet_filter_create(unit, 
                                       dest=[ KCOM_DEST_T_NETIF, netif_id, 0 ])
    print "netif %d filter %d" % ( netif_id, filter_id )


