#!/bin/bash
# Copyright (c) 2015, 2021 by cisco Systems, Inc.
# January 2015, Wilson Talaugon
#
#-----------------------------------------------------------------------
# tmpdir_cleanup.sh:
#     This script is responsible for monitoring the disk usage in 
#     /tmp directory and cleaning up subdirectories and files.
#
# Arguments
# $1 - /tmp dir disk usage in % before the cleanup script starts deleting
#      subdirectories and files.
#

declare -r TMPDIR=/tmp
declare -r ARCHIVE_DIR=/misc/disk1
declare -r TMP_COPY=tmp_archive
declare -r BLACKLIST_FILE=/etc/tmpdir_cleanup.blacklist
declare -r TRUE=0
declare -r FALSE=1
declare -i CLEAN_OPER=0
declare -r RM_NONE=0
declare -r RM_ALL=1
declare -r RM_CONTENT=2
declare -a BLACKLIST
declare -a TMPFILES
declare -r TMP_USAGE=$1

# Script function to read the tmp blacklist file and 
# load the content into BLACKLIST global array.
function get_blacklist
{
   local line=""
   local line_element=""

   # Load blacklist file into array
   let i=0
   while IFS=$'\n' read -r line; do
       line_element=`echo ${line}|tr -d " \t\n"`
       [[ $line_element = \#* ]] && continue
       [[ -z "$line_element" ]] && continue
       BLACKLIST[i]=$line_element
       ((++i))
   done < $BLACKLIST_FILE
}

# Script function to check if a given file or dir is in the blacklist
# file and sets CLEAN_OPER with the type of cleanup operation.
# RM_ALL - delete everything including its subdirectories
# RM_CONTENT - remove only its content including subdirectories
# RM_NONE - don't remove the file or directory
function check_clean_oper
{
   local file=$1
   local match=""
   CLEAN_OPER=0
   if [ -d $TMPDIR/$file ]; then
      match=`printf "%s\n" "${BLACKLIST[@]}" | grep -x "$file\/\*"`
      if [[ -n $match ]]; then
         CLEAN_OPER=$RM_CONTENT
         return
      fi
   fi
   match=`printf "%s\n" "${BLACKLIST[@]}" | grep -x "$file"`
   if [[ -z $match ]]; then
      CLEAN_OPER=$RM_ALL
      return
   fi
}

# Script file to archive deleted tmp files.
function copy_deleted_files
{
   local dir_file=$1
   if [ -d $ARCHIVE_DIR ]; then
      if [[ ! -d $ARCHIVE_DIR/$TMP_COPY ]]; then
         mkdir -p $ARCHIVE_DIR/$TMP_COPY 
      fi
      cp -af --parents $dir_file $ARCHIVE_DIR/$TMP_COPY
   fi
}

# Script function to clean up the tmp directory.
# It will check first if the file or directory is in the
# blacklist file before removing it. Message will be 
# logged to /var/log/messages.
function cleanup_tmp_dir
{
   local -a tmpfiles
   # Load tmp directory listing into the local array
   tmpfiles=($(ls -1 $TMPDIR))

   # Load blacklist files into the global array
   get_blacklist

   let i=0
   /usr/bin/logger -t TMPDIR "/tmp at $TMP_USAGE%. Cleaning up files and may"\
           " be archived in /misc/disk1/tmp_archive." 
   # CSCvw99520 Limit /tmp copy to /tmp_archive to avoid consuming harddisk space
   if [ -d $ARCHIVE_DIR/$TMP_COPY ]; then
       /usr/bin/logger -t TMPDIR "$ARCHIVE_DIR/$TMP_COPY contents are removed"\
           " before copying $TMPDIR to $ARCHIVE_DIR/$TMP_COPY."
       rm -rf $ARCHIVE_DIR/$TMP_COPY/*
   fi
   while (( ${#tmpfiles[@]} > i )); do
      check_clean_oper ${tmpfiles[i]}
      if [ $CLEAN_OPER -eq $RM_ALL ]; then
          copy_deleted_files $TMPDIR/${tmpfiles[i]}
          rm -rf $TMPDIR/${tmpfiles[i]}
      elif [ $CLEAN_OPER -eq $RM_CONTENT ]; then
          copy_deleted_files $TMPDIR/${tmpfiles[i]}
          rm -rf $TMPDIR/${tmpfiles[i]}/*
      fi
      ((++i))
   done
}

# Script function to check if the tmp directory is filled
# with the given threshold (%).
# TRUE - matched or exceeded the given threshold
# FALSE - below threshold
function is_tmp_dir_filled
{
   local usage_col=`df -h $TMPDIR | awk '{for(i=1;i<NF;i++) {if($i ~ /Use%/)\
         { print i }}}'`
   local usage=`df -h $TMPDIR | awk '{if (NR!=1) {print $'$usage_col'}}' |\
         sed "s/%//"`
   if [ "$usage" -ge "$TMP_USAGE" ]; then
      return $TRUE
   else
      return $FALSE
   fi
}

case "$1" in
 status)
    pid=(`pidof -x $0`)
    if [ ${#pid[@]}  -gt 1 ]; then
        echo "$0 (pid $pid) is running ..."
    else
        echo "$0 is not running"
    fi
    exit 0
    ;;
esac


# Check the tmp dir usage (%) before cleaning up the directory
is_tmp_dir_filled && cleanup_tmp_dir
