#!/bin/bash 
# 
# lxc - libvirt pd hook script for platform zermatt
# This script is added as a pd hack to pass no host interface from host to XR
# container for zermatt
#
# Copyright (c) 2014-2021 by Cisco Systems, Inc.

. /etc/init.d/functions
. /etc/init.d/spirit-functions # populate_rootfs_vars
. /etc/init.d/pd-functions

. /etc/init.d/spirit_pd.sh
#platform_hw_profile_get_settings

# Netflow vlans go to LC
lc_intf_vlan=(
         "ps-inb.1282"    #0x502 Netflow VLAN for NPU 1
         "ps-inb.1298"    #0x512 Netflow VLAN for NPU 2 
         "ps-inb.1314"    #0x522 Netflow VLAN for NPU 3
         "ps-inb.1330"    #0x532 Netflow VLAN for NPU 4
         "ps-inb.1346"    #0x542 Netflow VLAN for NPU 5
         "ps-inb.1362"    #0x552 Netflow VLAN for NPU 6
         "ps-inb.1378"    #0x562 Netflow VLAN for NPU 7
         "ps-inb.1394"    #0x572 Netflow VLAN for NPU 8
         "ps-inb.1570"    #0x622 OLP VLAN
	 "ps-inb.1554"    #0x612 BFD VLAN
    )
#Netflow VLAN for 2J LCs with PCIe based inband
lc_2j_intf_vlan=(
         "ps-inb0.1282"    #0x502 Netflow VLAN for NPU 1
         "ps-inb1.1298"    #0x512 Netflow VLAN for NPU 2 
    )

lc_1j_intf_vlan=(
         "ps-inb0.1282"    #0x502 Netflow VLAN for NPU 1
    )

#Netflow VLAN for 4J LCs with PCIe based inband
lc_4j_intf_vlan=(
         "ps-inb0.1282"    #0x502 Netflow VLAN for NPU 1
         "ps-inb1.1298"    #0x512 Netflow VLAN for NPU 2 
         "ps-inb2.1314"    #0x522 Netflow VLAN for NPU 3
         "ps-inb3.1330"    #0x532 Netflow VLAN for NPU 4
    )

# Netstack vlan to RP
rp_intf_vlan=( 
         "ps-inb.1794"
         "ps-inb.1778"
    )

lc_1j_intf_pcie_inb_vlan=(
         "ps-inb0.1538"    
         "ps-inb0.1554"   
         "ps-inb0.1570"   
         "ps-inb0.1794"   
         "ps-inb0.1778"   
    )
lc_2j_intf_pcie_inb_vlan=(
         "ps-inb0.1538"    
         "ps-inb0.1554"   
         "ps-inb0.1570"   
         "ps-inb0.1794"   
         "ps-inb0.1778"   
         "ps-inb1.1538"    
         "ps-inb1.1554"   
         "ps-inb1.1570"   
         "ps-inb1.1794"   
         "ps-inb1.1778"   
    )

lc_4j_intf_pcie_inb_vlan=(
         "ps-inb0.1538"    
         "ps-inb0.1554"   
         "ps-inb0.1570"   
         "ps-inb0.1794"   
         "ps-inb0.1778"   
         "ps-inb1.1538"    
         "ps-inb1.1554"   
         "ps-inb1.1570"   
         "ps-inb1.1794"   
         "ps-inb1.1778"   
         "ps-inb2.1538"    
         "ps-inb2.1554"   
         "ps-inb2.1570"   
         "ps-inb2.1794"   
         "ps-inb2.1778"   
         "ps-inb3.1538"    
         "ps-inb3.1554"   
         "ps-inb3.1570"   
         "ps-inb3.1794"   
         "ps-inb3.1778"   
    )
#invoked with $1 $2 $3 $4 as received by the caller
function pd_handle_init_start()
{
   local vm_name=$1
   local func_exit_code=0

   case "$vm_name" in
    sysadmin)
        lxc_set_sysadmin_cpuset_hook $1 $2 $3 $4
	func_exit_code=$?
        ;;

    default-sdr--*)
        lxc_set_sdr_cpuset_hook $1 $2 $3 $4
	func_exit_code=$?
        ;;

    *)
        #Nothing to do
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_handle_init_start

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_set_sysadmin_cpuset_hook()
{
    local vm_name=$1
    if [ -d /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc ]; then
        log_msg "in lxc_set_sysadmin_cpuset_hook cp_cores $cp_cores"
        echo $cp_cores > /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc/cpuset..cpus
    else
        log_msg "in lxc_set_sysadmin_cpuset_hook cgroup dir not set"
        return 1
    fi

    return 0
}
readonly -f lxc_set_sysadmin_cpuset_hook

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_reclaim_sysadmin_intf_hook()
{
    local vm_name=$1
    log_msg "Nothing to do for VM $vm_name"

    return 0
}
readonly -f lxc_reclaim_sysadmin_intf_hook

#
# invoked with $1 $2 $3 $4 as received by the caller
#
# $1 : Name of the LXC, used in the XML.
#      eg: sysadmin/default-sdr--1
# $2 : Operation of the LXC.
#      (prepare/start/init_start/started/stopped/release/reconnect)
# $3 : Sub-operation of the LXC mentioned in $2
#      (begin/end)
# $4 : One additional argument.
#      For operation init_start, we get the PID of /sbin/init of container.
#
function lxc_reclaim_sdr_intf_hook()
{
    return 0
}

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_set_sdr_cpuset_hook()
{
    local vm_name=$1
    if [ -d /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc ]; then
        log_msg "in lxc_set_sdr_cpuset_hook cp_cores $cp_cores"
        echo $cp_cores > /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc/cpuset..cpus
    else
        log_msg "in lxc_set_sdr_cpuset_hook cgroup dir not set"
        return 1
    fi
  
    return 0
}
readonly -f lxc_set_sdr_cpuset_hook

function lxc_set_sdr_intf_lc_hook()
{
    return 0
}

function lxc_set_sdr_intf_rp_hook()
{
    local namespace=$1

    nproc=`nproc`
    if [ "$nproc" -ge "12" ]; then
        cpumask="f00"
    elif [ "$nproc" -ge "8" ]; then
        cpumask="0f0"
    else
        cpumask="00f"
    fi

    echo $cpumask > /sys/devices/virtual/net/xr-eth-rp-spp/queues/rx-0/rps_cpus 
    echo $cpumask > /sys/devices/virtual/net/xr-eth-rp-spp/queues/tx-0/xps_cpus 
}

# Below platfoms only has single management interface under fretta fixed
# platform family
# 27027 # Tortin (NCS5501-HD)
# 27029 # Pyke (NCS-55A1-24H)
# 27030 # Peyto(NCS5502-MOD-SE-S) 
# 27031 # Peyto(NCS5502-MOD-S) 
# 27045 # Peyto(NCS5502-MOD-HD-S) 
# 60001 # Peyto(NC55A2-MOD-SE-H-S)
# 27047 # Bifrost-T (NCS-55A1-48Q6H) 
# 60004 # Turin-CR (NCS-55A1-48T6H) 
function lxc_set_sysadmin_intf_hook()
{
    local card_index=$(get_cmdline_card_index)
    case ${card_index} in 
        27027 |  27029 | 27030 | 27031 | 27045 | 60001 | 60004 | 27047 )
          log_msg "lxc_set_sysadmin_intf_hook: No eth-adm-mgmt, return"
          return 0
          ;;
    *)
      ;;
    esac

    local namespace=$4
    log_msg "lxc_set_sysadmin_intf_hook: namespace $namespace"
    ip link set eth-adm-mgmt netns $namespace

}
readonly -f lxc_set_sysadmin_intf_hook

#
# pass interfaces to xr, invoked with $1 $2 $3 $4 as received by caller
#
# $1 : Name of the LXC, used in the XML.
#      eg: sysadmin/default-sdr--1
# $2 : Operation of the LXC.
#      (prepare/start/init_start/started/stopped/release/reconnect)
# $3 : Sub-operation of the LXC mentioned in $2
#      (begin/end)
# $4 : One additional argument.
#      For operation init_start, we get the PID of /sbin/init of container.
#
function lxc_set_sdr_intf_hook()
{
    local vm_name=$1
    case "$vm_name" in
    default-sdr--1)
        lxc_set_sdr_intf_rp_hook $4
        return 0
        ;;
    default-sdr--2)
        lxc_set_sdr_intf_lc_hook $4
        return 0
        ;;
    *)
        return 1
        ;;
    esac
}

#
# pd_unmount_lxc_rootfs
#   PI counter part is present in lxc_hook.sh
#
function pd_unmount_lxc_rootfs ()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    sysadmin|default-sdr--*)

        unmount_lxc_rootfs "$vm_name"
        func_exit_code=$?
        ;;

    *)
        log_err "pd_unmount_lxc_rootfs: VM $vm_name, not supported"
        ;;

    esac

    return $func_exit_code
}
readonly -f pd_unmount_lxc_rootfs

JERICHO_CNTRL_REG_0=0x000003A0
JERICHO_CNTRL_REG_1=0x000003A4
DISCONNECT_JERICHO=0x8080808
GREEN_MODE_POWER_ENABLE=0x000003F8
JERICHO_POWER_ENABLE=0xF
FE3200_CNTRL_REG=0x000003F0
IOFPGA_RESET_CTRL_REG=0x00000084
RESET_BIT_BCM8456=6
RESET_BIT_BCM88375=4
OC_APM_OPTICS_ENABLE_BITS=0xF00C00

function wait_for_devices_poweroff()
{
    # 5 sec delay is required for succesful power-off of slice
    # devices before powering them back up.
    sleep 5 
}

IOFPGA_HOST_EVENT_REG_ADDR=0x18
DM2_PRESHUTDOWN_NOTIF=1
# Steps to bring down the device
# 0. Notify card_mgr about slices power going down
# 1. rmmod of the kernel module
# 2. remove the devices from the pci heirarchy
# 3. Power down the device
function lxc_xr_lc_disable_dnx_devices()
{
    #
    # Before turning off the power domain 2, we need to notify card_mgr
    # via scratch register about this, so sensors on DM2 that is currently
    # being monitored could be stopped to prevent getting bogus reading
    # when the power is turned off.
    # We will give card_mgr 2 seconds timeout to process this message.
    local retry=0
    local reg_val=$DM2_PRESHUTDOWN_NOTIF
    log_msg "Notifying card_mgr about DM2 pre-shutdown event ..."
    iofpga_reg_write 0 $IOFPGA_HOST_EVENT_REG_ADDR $reg_val
    until [ $reg_val -eq 0 ]; do
        reg_val=$(iofpga_reg_read 0 $IOFPGA_HOST_EVENT_NOTIF_REG_ADDR)
        reg_val=$(($reg_val & 0xF))
        if [ $reg_val -ne 0 ]; then
            retry=$((retry + 1))
            if [ $retry -gt 20 ]; then
                log_err "Timeout waiting for card_mgr to respond to DM2 pre-shutdown notification. Procced Forced Shutdown."
                break;
            fi
            sleep 0.1
        else
            log_msg "card_mgr responded to DM2 pre-shutdown notification in $((retry * 100)) ms"
        fi
    done

    # Find out card index so PCI base address for the 
    # respective boards 
    # BASE BOARD and Mezz board 
    # 27014 = Zermatt(NCS5502-SE)       8 Jericho Version
    # 27015 = Zermatt(NCS5502-SE-PROTO) 2 Jericho Version
    # 27016 = Turin(NCS5501-SE)         1 Qumran  Version
    # 27019 = Zermatt(NCS5502)          8 Jericho Version, no TCAM
    # 27020 = Taihu(NCS5501)            1 Qumran  Version
    # 27021 = Winterfell(NCS5501-A2-SE) 2 J+ Version
    # 27025 = Oldcastle(NCS5501-A1-SE)  4 J+ Version
    # 27026 = Oldcastle-CR(NCS5501-A1)  4 J+ Version, no TCAM
    # 27027 = Tortin(NCS5501-HD)        1 Qumran Version, no TCAM
    # 27029 = Pyke(NCS-55A1-24H)          2 J+ Version, no TCAM
    # 27030 = Peyto(NCS5502-MOD-SE-S)   1 J+Version, with TCAM 
    # 27031 = Peyto(NCS5502-MOD-S)      1 J+Version, no  TCAM 
    # 27045 = Peyto(NCS5502-MOD-HD-S)   1 J+Version, no  TCAM  HD
    # 60001 = Peyto(NC55A2-MOD-SE-H-S)  1 J+Version, with TCAM HD CC
    # 27047 = Bifrost-T (NCS-55A1-48Q6H) 2 J+Version, no TCAM
    # 60004 = Turin-CR (NCS-55A1-48T6H) 1 J+Version, no TCAM
    
    local card_index=$(get_cmdline_card_index)
    case ${card_index} in 
        27014 | 27015 | 27019)
               for bdf in $(lspci -nn | grep "14e4:86" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # Assert slice disconnect signal before putting slices in reset
               # for slices in both Base and Mezz IOFPGA

               # BASE: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_0)
               reg_val=$(($reg_val | 0x8080808))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_1)
               reg_val=$(($reg_val | 0x808))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_1 $(printf 0x%08X $reg_val)

               # Mezz: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 2 $JERICHO_CNTRL_REG_0)
               reg_val=$(($reg_val | 0x8080808))
               iofpga_reg_write 2 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)
               reg_val=$(iofpga_reg_read 2 $JERICHO_CNTRL_REG_1)
               reg_val=$(($reg_val | 0x808))
               iofpga_reg_write 2 $JERICHO_CNTRL_REG_1 $(printf 0x%08X $reg_val)
               sleep 0.1

               # Base: put all slices in reset
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 0x0
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_1 0x0

               # Mezz: put all slices in reset
               iofpga_reg_write 2 $JERICHO_CNTRL_REG_0 0x0
               iofpga_reg_write 2 $JERICHO_CNTRL_REG_1 0x0

               # Turnn off slices power from both Base and Mezz IOFPGA
               iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE 0x0
               iofpga_reg_write 2 $GREEN_MODE_POWER_ENABLE 0x0
               wait_for_devices_poweroff

           ;;
        27020)
               # Bring down the pcie ethernet interface before removing the device
               ifconfig ps-inb down
               ifconfig bcm-04 down
	       
               # Delay to ensure knet interface cleanup before device removal
               sleep 3
           ;&            
        27016 | 27027)
               for bdf in $(lspci -nn | grep "14e4:8[3-4]" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # Put Qumran in reset
               reg_val=$(iofpga_reg_read 1 $IOFPGA_RESET_CTRL_REG)
               reg_val=$(($reg_val | (1 << $RESET_BIT_BCM88375)))
               iofpga_reg_write 1 $IOFPGA_RESET_CTRL_REG  $(printf 0x%08X $reg_val)

               # There is no support to turn off only power for Qumran
           ;;

        27029) ;& #Pyke SHKL_TBD assume same PCIbus as winterfell, fixme if diff
        27021)    #Winterfell
               ifconfig ps-inb0 down
               ifconfig bcm-06 down
               ifconfig ps-inb1 down
               ifconfig bcm-08 down

               sleep 3

               # Remove J+
               for bdf in $(lspci -nn | grep "14e4:86" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # Remove OP TCAM
               for bdf in $(lspci -nn | grep "14e4:98" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # BASE: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_0)
               reg_val=$(( $reg_val | $DISCONNECT_JERICHO ))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)

               sleep 0.1

               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 0x0

               #clear out the lowest nibble corresponding to J+
               reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
               reg_val=$(( $reg_val & ~$JERICHO_POWER_ENABLE )) 
               iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)

               wait_for_devices_poweroff 
            ;;

        27025 | 27026)  #oldcastle
               ifconfig ps-inb0 down
               ifconfig bcm-07 down
               ifconfig ps-inb1 down
               ifconfig bcm-09 down
               ifconfig ps-inb2 down
               ifconfig bcm-0c down
               ifconfig ps-inb3 down
               ifconfig bcm-0e down

               # Delay to ensure knet interface cleanup before device removal
               sleep 3
               
               for bdf in $(lspci -nn | grep "14e4:86" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # Remove OP TCAM
               for bdf in $(lspci -nn | grep "14e4:98" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # BASE: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_0)
               reg_val=$(( $reg_val | $DISCONNECT_JERICHO ))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)

               sleep 0.1

               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 0x0

               #clear out the lowest nibble corresponding to J+
               reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
               reg_val=$(( $reg_val & ~$JERICHO_POWER_ENABLE ))
               iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)

               wait_for_devices_poweroff
            ;;

        27030 | 27031 | 27045 | 60001) #Peyto 
               ifconfig ps-inb0 down
               ifconfig bcm-02 down
               
               # Delay to ensure knet interface cleanup before device removal
               sleep 3
               
               # Remove J+
               for bdf in $(lspci -nn | grep "14e4:86" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # Remove OP TCAM 
               for bdf in $(lspci -nn | grep "14e4:98" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # BASE: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_0)
               reg_val=$(( $reg_val | $DISCONNECT_JERICHO ))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)

               sleep 0.1

               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 0x0

               #clear out the lowest nibble corresponding to J+
               reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
               reg_val=$(( $reg_val & ~$JERICHO_POWER_ENABLE )) 
               iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)

               wait_for_devices_poweroff
           ;;

        27047)    #Bifrost-T 
               ifconfig ps-inb0 down
               ifconfig bcm-06 down
               ifconfig ps-inb1 down
               ifconfig bcm-08 down

               sleep 3

               # Remove J+
               for bdf in $(lspci -nn | grep "14e4:86" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # BASE: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_0)
               reg_val=$(( $reg_val | $DISCONNECT_JERICHO ))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)

               sleep 0.1

               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 0x0

               #clear out the lowest nibble corresponding to J+
               reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
               reg_val=$(( $reg_val & ~$JERICHO_POWER_ENABLE )) 
               iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)

               wait_for_devices_poweroff
            ;;

        60004)    # Turin-CR 
               ifconfig ps-inb0 down
               ifconfig bcm-04 down

               # Delay to ensure knet interface cleanup before device removal
               sleep 3
               
               # Remove J+
               for bdf in $(lspci -nn | grep "14e4:86" | cut -d " " -f 1);
               do
                   echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
               done

               # BASE: Assert slice disconnect signal
               reg_val=$(iofpga_reg_read 1 $JERICHO_CNTRL_REG_0)
               reg_val=$(( $reg_val | $DISCONNECT_JERICHO ))
               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 $(printf 0x%08X $reg_val)

               sleep 0.1

               iofpga_reg_write 1 $JERICHO_CNTRL_REG_0 0x0

               #clear out the lowest nibble corresponding to J+
               reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
               reg_val=$(( $reg_val & ~$JERICHO_POWER_ENABLE )) 
               iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)

               wait_for_devices_poweroff
            ;;

        *) echo "Unknown platform"
           log_err "Unknown platform $card_index"
           return -1
           ;;
    esac

    return 0
}

# Steps to bring down the  MPA devices 
# 1. From MIFPGA BAR 3 disable to power to both MPA's 
# 2. Once the power is disabled remove the MPA's from the
#    pci tree
# NOTE: Once the LX LXC boot's up it will power on and then do
#       pciscan to bring the MPA FPGA devices back
# Disable for following boards:
# 1) NCS-55A2-MOD-S    (FPGA ID = 0xb5171fd3)
# 2) NCS-55A2-MOD-HD-S (FPGA ID = 0xd5171fd3)
# 3) NCS-55A2-MOD-SE-S (FPGA ID = 0xa5171fd3)
# 4) NCS-55A2-MOD-SE-H-S (FPGA ID = 0xf5171fd3)
function lxc_xr_lc_disable_mpafpga_devices()
{
    local platform_fpga_id
    local f_id

    platform_fpga_id=$(mifpga_reg_read 0 0x4)
    f_id=`echo "$platform_fpga_id" | awk -F' ' '{print $1}'`

    log_msg "Entered disable mpafpga for platform id $f_id"

    case "$f_id" in
        0xb5171fd3 | 0xa5171fd3 | 0xd5171fd3 | 0xf5171fd3)
                   # power off MPA power enable register 0x14 
                   log_msg "Turning off power Write 0x0 to 0x14 MPA CTRL BAR"
                   mifpga_reg_write 3 0x14 0x0

                   if [ -f /sys/bus/pci/devices/0000:05:00.0/remove ]; then
                       log_msg "MPA1 remove path found removing"
                       echo 1 > /sys/bus/pci/devices/0000:05:00.0/remove
                   else
                       log_msg "MPA1 removal path not found"
                   fi
                   if [ -f /sys/bus/pci/devices/0000:07:00.0/remove ]; then
                       log_msg "MPA2 remove path found removing"
                       echo 1 > /sys/bus/pci/devices/0000:07:00.0/remove
                   else
                       log_msg "MPA2 removal path not found"
                   fi
                   ;;
        *)log_msg "Skipping $f_id for MPA device removal"
     esac

    return 0
}

function pd_handle_release ()
{
    local vm_name=$1
    local func_exit_code=0


    case "$vm_name" in
    sysadmin)
        log_msg "pd_handle_release: Nothing to do for VM $vm_name"
        ;;

    default-sdr--1)
        log_msg "pd_handle_release: Nothing to do for VM $vm_name"
        ;;

    default-sdr--2)
        log_msg "pd_handle_release: VM $vm_name"
        # On LXC down we need to power down and remove the MPA
        # device. This will apply to only to MPA based platforms 
        # we want to turn of MPA power down before DM2 power in 
        # lxc_xr_lc_disable_dnx_devices goes down and remove 
        # PCI MPA FPGA devices
        # eg - PeytoCr, PeytoCrHD, PeytoSE 
        log_msg "calling lxc_xr_lc_disable_mpafpga_devices"
        lxc_xr_lc_disable_mpafpga_devices

        log_msg "calling lxc_xr_lc_disable_dnx_devices"
        lxc_xr_lc_disable_dnx_devices
        func_exit_code=$?
        ;;

    *)
        log_err "pd_handle_release: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_handle_release

# Important note: 
# For Zermatt(NCS-5502) system the Jerichos that are used in IOFPGA registers
# are not same bit alignment. For zermatt we would be using following numbers
# Jericho 1,3,4,5. When we enable the devices via the IOFPGA write operation
# we would need to take care we touch the correct bits for Jer 1,3,4,5
#
function lxc_xr_lc_enable_dnx_devices()
{

    # write to the sctrach register to
    # notify the card_mgr about starting of XR VM.
    local STARTING_XR_VM_NOTIF=2
    iofpga_reg_write 0 $IOFPGA_HOST_EVENT_REG_ADDR $STARTING_XR_VM_NOTIF

    # This delay is required to make sure that down interrupt
    # is handled propelry before enabling the power
    sleep 7

    # Find out card index so PCI base address for the 
    # respective boards 
    # BASE BOARD 
    # 27014 = Zermatt(NCS5502-SE)       8 Jericho Version
    # 27015 = Zermatt(NCS5502-SE-PROTO) 2 Jericho Version
    # 27016 = Turin(NCS5501-SE)         1 Qumran  Version
    # 27019 = Zermatt(NCS5502)          8 Jericho Version, no TCAM
    # 27020 = Taihu(NCS5501)            1 Qumran  Version
    # 27021 = Winterfell(NCS5501-A2-SE) 2 J+ Version
    # 27025 = Oldcastle(NCS5501-A1-SE)  4 J+ Version
    # 27026 = Oldcastle-CR(NCS5501-A1)  4 J+ Version, no TCAM
    # 27027 = Tortin(NCS5501-HD)        1 Qumran Version, no TCAM
    # 27029 = Pyke(NCS5501-A3)          2 J+ Version, no TCAM
    # 27030 = Peyto(NCS5502-MOD-SE-S)   1 J+Version, with TCAM 
    # 27031 = Peyto(NCS5502-MOD-S)      1 J+Version, no  TCAM 
    # 27045 = Peyto(NCS5502-MOD-HD-S)   1 J+Version, no  TCAM  HD
    # 60001 = Peyto(NC55A2-MOD-SE-H-S)  1 J+Version, with TCAM HD CC
    # 27047 = Bifrost-T (NCS-55A1-48Q6H) 2 J+Version, no TCAM
    # 60004 = Turin-CR (NCS-55A1-48T6H) 1 J+Version, no TCAM
    
    local card_index=$(get_cmdline_card_index)
    case ${card_index} in 
        27014 | 27015 | 27019)
            # Handling of enabling slices power and take them out of reset has been
            # moved to card manager & slice manager plugin running from SysAdmin VM.
            return 0
           ;;

        27016 | 27020 | 27027)
               lspci_port="14e4:8[3-4]"     # Turin

               # take Qumran out of reset
               reg_val=$(iofpga_reg_read 1 $IOFPGA_RESET_CTRL_REG)
               reg_val=$(( $reg_val & ~(1 << $RESET_BIT_BCM88375) ))
               iofpga_reg_write 1 $IOFPGA_RESET_CTRL_REG  $(printf 0x%08X $reg_val)
           ;;

        27021)
              reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
              reg_val=$(( $reg_val | 0x1C))
              iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)
              return 0
           ;;

        27029)
              reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
              reg_val=$(( $reg_val | 0x300003))  
              iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)
              return 0
           ;;

        27030 | 27031 | 27045 | 60001)
              reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
              reg_val=$(( $reg_val | 0x100400))
              # cross-check with FPGA spec. TBD
              iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)
              return 0
          ;;

        27025 | 27026)
              log_err "NCS-55A1-36H-SE-S and NCS-55A1-36H-S: Power on Optics, and APM.  right place to enable ES200 & optics pwr ?"
              reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
              reg_val=$(( $reg_val | $OC_APM_OPTICS_ENABLE_BITS ))  
              iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)
              return 0
           ;;

	27047)
              reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
              reg_val=$(( $reg_val | 0x300403))  
              iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)
              return 0
           ;;

	60004)
              reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
              reg_val=$(( $reg_val | 0x100300)) 
              iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)
              return 0
           ;;

        *) echo "Unknown platform"
           log_err "Unknown platform $card_index"
           return -1
           ;;
    esac

    # Add some sleep for devices to power up then we can do rescan
    sleep 5

    local num_devices=`lspci -nn | grep $lspci_port | cut -d " " -f 1 |wc -l`

    until [ $num_devices -gt 0 ]; do
        log_msg "rescanning dnx devices"
        case ${card_index} in 
        27016 | 27020 | 27027) 
            # Qumran Rescan 
            echo 1 > /sys/class/pci_bus/0000:04/rescan
           ;;
       
        *) 
           ;;
        esac
        num_devices=`lspci -nn | grep $lspci_port | cut -d " " -f 1 |wc -l`
        if [ $num_devices -le 0 ]; then
            sleep 0.5
        fi
    done

    case ${card_index} in 
        27020 )
               # Bring back ps-inb to life, this should be done
               # only after Qumran device is seen
               ifconfig bcm-04 up
               python /usr/sbin/bcm-knet-init.py start 0x04 ps-inb
               ifconfig ps-inb up
           ;;
       
        *)
               # Nothing to do for other platforms for now
           ;;
    esac

    return 0
}
readonly -f lxc_xr_lc_enable_dnx_devices

function lxc_xr_lc_enable_fe_devices()
{
    # 27014 = Zermatt(NCS5502-SE)    8 Jericho Version
    # 27015 = Zermatt(NCS5502-PROTO) 2 Jericho Version
    # 27016 = Turin(NCS5501-SE)      1 Qumran  Version
    # 27019 = Zermatt(NCS5502)       8 Jericho Version, no TCAM
    # 27020 = Taihu(NCS5501)         1 Qumran  Version
    # 27021 = Winterfell(NCS5501-A2-SE) 2 J+ Version
    # 27025 = Oldcastle(NCS5501-A1-SE)  4 J+ Version
    # 27026 = Oldcastle-CR(NCS5501-A1)  4 J+ Version, no TCAM
    # 27027 = Tortin(NCS5501-HD)        1 Qumran Version, no TCAM
    # 27029 = Pyke(NCS5501-A3)          2 J+ Version, no TCAM
    # 27030 = Peyto(NCS5502-MOD-SE-S)   1 J+Version, with TCAM 
    # 27031 = Peyto(NCS5502-MOD-S)      1 J+Version, no  TCAM 
    # 27045 = Peyto(NCS5502-MOD-HD-S)   1 J+Version, no  TCAM  HD
    # 60001 = Peyto(NC55A2-MOD-SE-H-S)  1 J+Version, with TCAM HD CC
    # 27047 = Bifrost-T (NCS-55A1-48Q6H) 2 J+Version, no TCAM
    # 60004 = Turin-CR (NCS-55A1-48T6H) 1 J+Version, no TCAM

    local card_index=$(get_cmdline_card_index)

    case ${card_index} in
        27014 | 27015 | 27019)
            #FE3600 out of reset Mezz board only
            iofpga_reg_write 2 FE3200_CNTRL_REG 0x10001
            ;;

        27016 | 27020 | 27021 | 27027 | 27029 | 27030 | 27031 | 27045 | 60004 | 60001 | 27047)
            return 0
            ;;

        27025 | 27026)
            #Power up the FE3600
            reg_val=$(iofpga_reg_read 1 $GREEN_MODE_POWER_ENABLE)
            reg_val=$(( $reg_val | 0x40000000 ))  
            iofpga_reg_write 1 $GREEN_MODE_POWER_ENABLE $(printf 0x%08X $reg_val)

            sleep 0.5

            #Take the FE out of reset
            iofpga_reg_write 1 FE3200_CNTRL_REG 0x1
            ;;

        *)
            echo "Unknown platform"
            log_err "Unknown platform $card_index"
            return -1
            ;;
    esac

    # Add some sleep for devices to come out of reset then we can do rescan
    sleep 1

    # FE3200: 14e4:8950
    # FE3600: 14e4:8775
    local num_devices=`lspci -nn | egrep "14e4:89|14e4:87" | cut -d " " -f 1 |wc -l`

    until [ $num_devices -gt 0 ]; do
        log_msg "rescanning fe devices"
        case ${card_index} in
            27025 | 27026)
                # FE 3600 Rescans
                echo 1 > /sys/class/pci_bus/0000:06/rescan
                ;;
            27014 | 27015 | 27019)
                # FE 3600 Rescans
                echo 1 > /sys/class/pci_bus/0000:0f/rescan
                echo 1 > /sys/class/pci_bus/0000:11/rescan
                ;;

            27016 | 27020)
                ;;

            *)
                ;;
        esac

        num_devices=`lspci -nn | egrep "14e4:89|14e4:87" | cut -d " " -f 1 |wc -l`
        if [ $num_devices -le 0 ]; then
            sleep 0.5
        fi
    done

    return 0
}
readonly -f lxc_xr_lc_enable_fe_devices

function validate_mgmt_intf() {
    local vendor_id=8086
    local device_id=1533
    local dbg_file=/var/log/eth-mgmt-failure.log

    if lspci -d $vendor_id:$device_id | grep I210; then
        log_msg "Management NIC is available"
        local mgmt_intf_cnt=$( ifconfig -a | grep eth-mgmt | grep -v grep | wc -l )

        if [ $mgmt_intf_cnt != 0 ]; then
             log_msg "Management Interface is created"
             return
        fi
    fi

    log_msg "Management interface validation failed"

    # We need to print some info to console, but the console may have been
    # given to VMs at this moment. So, we keep the debugs in a file, which
    # will be dumped from pdreboot program later.
    stdbuf -oL echo "Management interface validation failed" &> $dbg_file
    stdbuf -oL lspci -nn | grep I210 &>> $dbg_file
    stdbuf -oL ifconfig eth-mgmt &>> $dbg_file
    stdbuf -oL echo "Attempting recovery via resetting board" &>> $dbg_file
    log_msg "Attempting recovery via reloadin the board"
    sync
    reboot
}
readonly -f validate_mgmt_intf

function lxc_xr_lc_enable_switch_device()
{
    # 27014 = Zermatt(NCS5502-SE)       8 Jericho Version
    # 27015 = Zermatt(NCS5502-SE-PROTO) 2 Jericho Version
    # 27016 = Turin(NCS5501-SE)         1 Qumran  Version
    # 27019 = Zermatt(NCS5502)          8 Jericho Version, no TCAM
    # 27020 = Taihu(NCS5501)            1 Qumran  Version
    # 27021 = Winterfell(NCS5501-A2-SE) 2 J+ Version
    # 27025 = Oldcastle(NCS5501-A1-SE)  4 J+ Version
    # 27026 = Oldcastle-CR(NCS5501-A1)  4 J+ Version, no TCAM
    # 27027 = Tortin(NCS5501-HD)        1 Qumran Version, no TCAM
    # 27029 = Pyke(NCS5501-A3)          2 J+ Version, no TCAM
    # 27030 = Peyto(NCS5502-MOD-SE-S)   1 J+Version, with TCAM 
    # 27031 = Peyto(NCS5502-MOD-S)      1 J+Version, no  TCAM 
    # 27045 = Peyto(NCS5502-MOD-HD-S)   1 J+Version, no  TCAM  HD
    # 60001 = Peyto(NCS55A2-MOD-SE-H-S) 1 J+Version, with TCAM HD CC
    local card_index=$(get_cmdline_card_index)

    case ${card_index} in
        27014 | 27015 | 27019)
            # Take BCM 8456 switch out of reset - Base board only
            # Clear bit 6
            reg_val=$(iofpga_reg_read 1 IOFPGA_RESET_CTRL_REG)
            reg_val=$(( $reg_val & ~(1 << $RESET_BIT_BCM8456) ))
            hex_reg_val=$(echo $(printf 0x%08X $reg_val))
            iofpga_reg_write 1 IOFPGA_RESET_CTRL_REG $hex_reg_val
            ;;

        27016 | 27020 | 27021 | 27025 | 27026 | 27027 | 27029 | 27030 | 27031 | 27045 | 27047 | 60004 | 60001)
            return 0
            ;;

        *)
            echo "Unknown platform"
            log_err "Unknown platform $card_index"
            return -1
            ;;
    esac

    # Add some sleep for devices to come out of reset then we can do rescan
    sleep 1

    # BCM53456 switch: 14e4:8456
    local num_devices=`lspci -nn | grep "14e4:8456" | cut -d " " -f 1 |wc -l`

    until [ $num_devices -gt 0 ]; do
        log_msg "rescanning switch device"
        case ${card_index} in 
            27014 | 27015 | 27019)
                echo 1 > /sys/class/pci_bus/0000:0e/rescan
                ;;
            27016 | 27020)
                ;;
            *)
                ;;
        esac

        num_devices=`lspci -nn | grep "14e4:8456" | cut -d " " -f 1 |wc -l`

        if [ $num_devices -le 0 ]; then
            sleep 0.5
        fi
    done
    return 0
}
readonly -f lxc_xr_lc_enable_switch_device

#
# Small routine to check if the interface exists
#
function check_intf_exists ()
{
    if [ -d /sys/devices/virtual/net/$1 ]; then
        return 0
    fi
    return 1
}

#
# lxc_xr_add_intf
#   Routine to add vlan interfaces before XR is launched
#
function lxc_xr_add_intf ()
{
    local intf
    local vlanid
    local intf_list=("${!1}")

    for i in "${intf_list[@]}";
    do
        check_intf_exists $i
        if [ $? -ne 0 ]; then
            CMD="vconfig add ${i/./ }"
            printf "%s CMD: %s\n" "$timestamp" "$CMD" >> $LXC_HOOK_LOG_FILE
            declare -F platform_log &>/dev/null && platform_log "$CMD"

            platform_log_exec "$CMD"
            if [ $? -ne 0 ]; then
                platform_log_error "Failed CMD: $CMD"
                return 1
            fi
        fi
    done
}

#
# wait_for_intf
#  Routine to use as handshake method to identify once NPU UP
#  Once NPU UP and PCIE inband interfaces getting created then
#  it will break loop and proceed with the routine lxc_xr_Add_intf
#
function wait_for_intf()
{
    local MAX_RETRY=40
    local retry_count=0
    local intf_list=("${!1}")
    for i in "${intf_list[@]}";
    do
        while [ $retry_count -lt $MAX_RETRY ]
        do
            check_intf_exists $i
            if [ $? -ne 0 ]; then
                log_msg "Intf not up count:$retry_count i:$i"
                let retry_count+=1
                if [ $retry_count -ge $MAX_RETRY ]; then
                    log_msg "TIME's UP for waiting PCIE interface creation"
                fi
                sleep 1
            else
                break;
            fi
        done
    done
}

#
# pd_prelaunch_setup
#   PI counter part is present in lxc_hook.sh
#
function pd_prelaunch_setup ()
{
    local vm_name=$1
    local func_exit_code=0
    local card_index=$(get_cmdline_card_index)

    case "$vm_name" in
    sysadmin)
        case ${card_index} in
        60004)
            # Check whether mgmt-intf  is created or not
            # If not created, reload the system to recover  
            log_msg "Calling validate_mgmt_intf"
            validate_mgmt_intf
            ;;
        *)
            ;;
        esac

        log_msg "calling lxc_xr_lc_enable_switch_device"
        lxc_xr_lc_enable_switch_device
        func_exit_code=$?
        if [ $func_exit_code -ne 0 ]; then
            return $func_exit_code
        fi

        log_msg "calling lxc_xr_lc_enable_fe_devices"
        lxc_xr_lc_enable_fe_devices
        func_exit_code=$?
        if [ $func_exit_code -ne 0 ]; then
            return $func_exit_code
        fi
        ;;

    default-sdr--1)
        case ${card_index} in
        27014)  ;& #Zermatt(NCS5502-SE)       8 Jericho Version
        27015)  ;& #Zermatt(NCS5502-SE-PROTO) 2 Jericho Version
        27016)  ;& #Turin(NCS5501-SE)         1 Qumran  Version
        27019)  ;& #Zermatt(NCS5502)          8 Jericho Version, no TCAM
        27020)  ;& #Taihu
        27027)     #Tortin(NCS5501-HD)        1 Qumran Version, no TCAM
            lxc_xr_add_intf rp_intf_vlan[@]
            ;;
        *)
            ;;
        esac
        prelaunch_setup "$vm_name"
        func_exit_code=$?
        ;;

    default-sdr--2)
        log_msg "calling lxc_xr_lc_enable_dnx_devices"
        lxc_xr_lc_enable_dnx_devices
        func_exit_code=$?
        if [ $func_exit_code -ne 0 ]; then
            return $func_exit_code
        fi

        case ${card_index} in
        27029) ;& #Pyke
        27021) ;& #Winterfell
	27047)    #Bifrost-T
            wait_for_intf lc_2j_intf_pcie_inb_vlan[@]
            lxc_xr_add_intf lc_2j_intf_vlan[@]
            ;;

        27025)  ;&  #Oldcastle(NCS5501-A1-SE)  4 J+ Version
        27026)      #Oldcastle(NCS5501-A1)  4 J+ Version
            wait_for_intf lc_4j_intf_pcie_inb_vlan[@]
            lxc_xr_add_intf lc_4j_intf_vlan[@]
            ;;

        27030)  ;&  #Peyto 1 J+ Version (NCS-55A2-MOD-SE-S)
        27031)  ;&  #Peyto 1 CR J+ Version (NCS-55A2-MOD-S)
        27045)  ;&  #Peyto 1 HD J+ Version (NCS-55A2-MOD-HD-S)
        60001)  ;&  #Peyto 1 HD CC J+ Version (NC55A2-MOD-SE-H-S)
        60004)      #Turin-CR 1 J+ Version 
            wait_for_intf lc_1j_intf_pcie_inb_vlan[@]
            lxc_xr_add_intf lc_1j_intf_vlan[@]
            ;;

        *)
            lxc_xr_add_intf lc_intf_vlan[@]
            ;;
        esac

        prelaunch_setup "$vm_name"
        func_exit_code=$?
        ;;

    *)
        log_err "pd_prelaunch_setup: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_prelaunch_setup

#
# Pass bootstrap CLI file to XR, invoked with $1 $2 $3 $4 as received by caller
# Currently this handles regular config only; if a need to handle admin config
# arises then the code below needs to be replicated with
# 'iosxr_config_admin.txt'.
#
function lxc_cpy_bootstrap_cli()
{
    # $XR_PATH
    populate_rootfs_vars

    # Hardcoded bootstrap CLI name - it must be called
    # this on the provided ISO disk
    local CLI=iosxr_config.txt
    
    # Temporary stored location before copy to XR
    local IN=/mnt/iso
    
    # Final location of CVAC bootstrap CLI which on XR becomes
    # /etc/sysconfig/iosxr_config.txt
    local OUT=$XR_PATH/etc/sysconfig

    log_msg "CVAC: Starting scan for bootstrap CLI on all devices"
    log_msg "CVAC: Config file in location  $IN"
    log_msg "CVAC: Config file out location $OUT"
    
    mkdir -p $IN
    if [[ $? -ne 0 ]]; then
        log_err "CVAC: Failed CMD: mkdir -p $IN"
        exit 1
    fi

    # Scan and mount all /dev/sd*, /dev/hd* and /dev/vd* devices...
    # If successfully mounted, check for bootstrap CLI and copy it to XR
    for device in $( ls /dev ); do
        if [[ $device =~ sd* || $device =~ hd* || $device =~ vd* ]]; then
            mount /dev/$device $IN -o loop
            if [[ $? -eq 0 ]]; then
                # Succesfully mounted - now look for a bootstrap CLI file
                log_msg "CVAC: Mounted /dev/$device to $IN"
                if [[ -e $IN/$CLI ]]; then
                    # Found one, copy it to XR LXC to be parsed when XR comes up
                    log_msg "CVAC: Found bootstrap CLI: $IN/$CLI"
                    cp $IN/$CLI $OUT
                    if [[ $? -eq 0 ]]; then
                        log_msg "CVAC: Copied bootstrap CLI to $OUT/$CLI"
                    else
                        log_err "CVAC: Failed to copy bootstrap CLI: $IN/$CLI to $OUT/$CLI"
                    fi
                fi
                log_msg "CVAC: Unmount /dev/$device from $IN"
                umount $IN -d
            fi
        fi
    done

    return 0
}

#
# lxc_xr_mv_intf
#   Routine to mv interfaces back to host before XR is shut down
#
function lxc_xr_mv_intf ()
{
    local intf_list=("${!1}")
    local vm_name=$2
    local func_exit_code=0

    for i in "${intf_list[@]}";
    do
        # Skip existence test
        log_msg "ip netns exec $vm_name.libvirt ip link set $i netns 1"
        ip netns exec $vm_name.libvirt ip link set $i netns 1
        func_exit_code=$?
        log_msg "result of move cmd is $func_exit_code"
    done
}

#
# handle_release_mv_interfaces
#
# hook invoked from handle_release to move interfaces
# back to host after destroying the container
# This seems necessary when there is KIM/LCND interaction
#
# The hook in lxc_cleanup_helper binary moves the physical interfaces and
#  virtual interfaces underneath them (e.g.: eth0) back to host
# Currently, this moves the ps interface back to host on sdr destroy
#
# Always return 0
#
# FIX_ME
# We have to close cases when libvirt_lxc crashes while running this script
#
function handle_release_mv_interfaces
{
    local vm_name=$1
    local func_exit_code=0
    local card_index=$(get_cmdline_card_index)

    case "$vm_name" in
    default-sdr--2)
        case ${card_index} in
        27029) ;& #Pyke 2J+ SHKL_TBD assume same PCIbus as winterfell, fixme if diff
        27021) ;& #Winterfell 2J+
	27047)    #Bifrost-T 2J+
            lxc_xr_mv_intf lc_2j_intf_vlan[@] $1
            ;;

        27025) ;&  #Oldcastle(NCS5501-A1-SE)  4 J+ Version
        27026)     #Oldcastle(NCS5501-A1)  4 J+ Version
            lxc_xr_mv_intf lc_4j_intf_vlan[@] $1
            ;;

        27030) ;& #Peyto 1J+ TCAM (NCS-55A2-MOD-SE-S)
        27031) ;& #Peyto 1J+ No TCAM (NCS-55A2-MOD-S)
        27045) ;& #Peyto 1J+ No TCAM HD (NCS-55A2-MOD-HD-S)
        60001) ;& #Peyto 1J+ TCAM HD CC (NC55A2-MOD-SE-H-S)
        60004)    #Turin-CR 1J+
            lxc_xr_mv_intf lc_1j_intf_vlan[@] $1
            ;;


        *)
            lxc_xr_mv_intf lc_intf_vlan[@] $1
            ;;
        esac
        ;;

    default-sdr--1)
        case ${card_index} in
        27014)  ;& #Zermatt(NCS5502-SE)       8 Jericho Version
        27015)  ;& #Zermatt(NCS5502-SE-PROTO) 2 Jericho Version
        27016)  ;& #Turin(NCS5501-SE)         1 Qumran  Version
        27019)  ;& #Zermatt(NCS5502)          8 Jericho Version, no TCAM
        27020)  ;& #Taihu
        27027)  #Tortin(NCS5501-HD)        1 Qumran Version, no TCAM
            lxc_xr_mv_intf rp_intf_vlan[@] $1
            ;;
        *)
            ;;
        esac
        ;;
    *)
        #Nothing to do
        ;;
    esac

    return $func_exit_code
}

#
# pd_handle_sched_runtime
#
# Zermatt has two XR's running so divide sched runtime by two
#
function pd_handle_sched_runtime
{
    local vm_name=$1
    local source_file=$2
    local target_file=$3

    local value=$(cat $source_file)
    local valueby2=$((value/2))

    echo $valueby2 > "$target_file"

    return 0
}

readonly -f pd_handle_sched_runtime
