#!/bin/bash
#
# calvados_launch_common.sh
# Common scripts used by calvados_launch.sh.
#
# Copyright (c) 2014-2021 by cisco Systems, Inc.
# All rights reserved.
  
#default variables, to be replaced by pd 
CONSOLE_IPADDR="127.0.0.1"
TTYS0PORT=49998
TTYS1PORT=49999
HVC0PORT=50000
HVC1PORT=50001

BOOTSTRAP_FILE="/etc/init.d/calvados_bootstrap.cfg"
if [ -f $BOOTSTRAP_FILE ]; then
    . $BOOTSTRAP_FILE
else 
    VIRT_METHOD="vm"
fi

SPIRIT_FUNCTIONS="/etc/init.d/spirit-functions"
if [ -f $SPIRIT_FUNCTIONS ]; then
    . $SPIRIT_FUNCTIONS
fi

SPIRIT_PD="/etc/init.d/spirit_pd.sh"
if [ -f $SPIRIT_PD ]; then
    . $SPIRIT_PD
fi

get_board_type

#VIRT_METHOD is used in bootstrap.cfg file
#VIRT_ARCH is specifically defined for xml file

# Constants: host files to be used by Calvados install commit 
# Each file contains a single line in the format of:
# <BOOT_PARTITON>[:<DATA_PARTITION>]
# For instance:
# /dev/panini_vol_grp/calvados_lv0:/dev/panini_vol_grp/calvados_data_lv0
# The colon and data partition are optional

# Committed boot/data partion
readonly F_COMMITTED="/misc/config/calvados_launch_path.txt"
# Active boot/data partition
readonly F_ACTIVE="/misc/config/calvados_launch_path_active.txt"

# Backup of grub cfg and efi file path 
readonly GRUB_EFI_BACK="/misc/config/grub.efi"
readonly GRUB_CFG_BACK="/misc/config/grub.cfg"
readonly GRUB_MENU_LST_BACK="/misc/config/menu.lst"
readonly BZIMAGE_BACK="/misc/config/bzImage"
readonly INITRD_BACK="/misc/config/initrd.img"
readonly COMMITTED_MARKER_FILE="/install_comm/committed_partition.txt"
readonly NBI_BACK="/misc/config/sflc-auto.img"

function find_fs_offset {
    local start=$(fdisk -u -l ${1} 2> /dev/null | grep "${1}${2}" | awk '{print $2}')
    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} 2> /dev/null | grep "${1}${2}" | awk '{print $3}')
    fi
    local offset=$(expr ${start} \* 512)
    echo ${offset}
}
readonly -f find_fs_offset

function find_fs_sizelimit {
    local result
    local sizelimit
    local start=$(fdisk -u -l ${1} 2> /dev/null | grep "${1}${2}" | awk '{print $2}')
    local endlimit=$(fdisk -u -l ${1} 2> /dev/null | grep "${1}${2}" | awk '{print $3}')

    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} 2> /dev/null | grep "${1}${2}" | awk '{print $3}')
        endlimit=$(fdisk -u -l ${1} 2> /dev/null | grep "${1}${2}" | awk '{print $4}')
    fi
    result=$((${endlimit} - ${start}))
    sizelimit=$(expr ${result} \* 512)
    echo ${sizelimit}
}
readonly -f find_fs_sizelimit

function cal_cmn_build_domain_lxc_xml {
cat << end_build_domain_lxc_xml >> $OUT_XML_FILE
<domain type='lxc' id='2' xmlns:lxc='http://libvirt.org/schemas/domain/lxc/1.0'>
end_build_domain_lxc_xml
}

function cal_cmn_build_domain_kvm_xml {
cat << end_build_domain_kvm_xml >> $OUT_XML_FILE
<domain type='kvm' xmlns:qemu='http://libvirt.org/schemas/domain/qemu/1.0'>
end_build_domain_kvm_xml
}

function cal_cmn_build_domain_end_xml {
cat << end_build_domain_end_xml >> $OUT_XML_FILE
</domain>
end_build_domain_end_xml
}

function cal_cmn_build_os_lxc_xml {
cat << end_build_os_lxc_xml >> $OUT_XML_FILE
  <os>
    <type>exe</type>
    <init>/sbin/init</init>
  </os>
end_build_os_lxc_xml
}

function cal_cmn_build_os_kvm_xml {
cat << end_build_os_kvm_xml >> $OUT_XML_FILE
  <os>
    <type arch='x86_64' machine='pc-q35-1.5'>hvm</type>
    <boot dev='hd'/>
  </os>
end_build_os_kvm_xml
}

function cal_cmn_build_virt_base_cfg_xml {

    if [ "$VIRT_ARCH" == "lxc" ]; then
        cal_cmn_build_domain_lxc_xml
    elif [ "$VIRT_ARCH" == "qemu" ]; then
        cal_cmn_build_domain_kvm_xml
    else 
        declare -F platform_log &>/dev/null && platform_log \
            "$FUNCNAME: *ERROR* invalid VM MODE $VIRT_ARCH"
        exit
    fi

cat << end_build_virt_part1_xml >> $OUT_XML_FILE
  <name>$VMNAME</name>
  <memory unit='KiB'>$VIRT_MEM</memory>
  <currentMemory unit='KiB'>$VIRT_MEM</currentMemory>
  <vcpu placement='static'>$VIRT_SMP</vcpu>
  <features>
    <acpi/>
    <pae/>
  </features>
  <clock offset='utc'/>
  <on_poweroff>destroy</on_poweroff>
  <on_reboot>restart</on_reboot>
  <on_crash>destroy</on_crash>
end_build_virt_part1_xml
}

function cal_cmn_build_devices_start_xml {
cat << end_build_devices_start_xml >> $OUT_XML_FILE
  <devices>
end_build_devices_start_xml
}

function cal_cmn_build_devices_end_xml {
cat << end_build_devices_end_xml >> $OUT_XML_FILE
  </devices>
end_build_devices_end_xml
}

function cal_cmn_build_disk_simple_xml {
cat << end_build_disk_xml >> $OUT_XML_FILE
    <disk type='block' device='disk'>
      <driver cache='none'/>
      <source dev='$DRIVE_SRC_DEV'/>
      <target dev='$DRIVE_TGT_DEV' bus='virtio'/>
      <alias name='$ALIAS_NAME'/>
    </disk>
end_build_disk_xml
}

function cal_cmn_build_disk_xml {
cat << end_build_disk_xml >> $OUT_XML_FILE
    <disk type='block' device='disk'>
      <driver name='$VIRT_ARCH' type='raw' cache='none'/>
      <source dev='$DRIVE_SRC_DEV'/>
      <target dev='$DRIVE_TGT_DEV' bus='virtio'/>
      <alias name='$ALIAS_NAME'/>
    </disk>
end_build_disk_xml
}

function cal_cmn_build_usb_xml {
cat << end_build_usb_xml >> $OUT_XML_FILE
    <controller type='usb' index='0'/>
    <controller type='sata' index='0'/>
    <controller type='pci' index='0' model='pcie-root'/>
    <controller type='pci' index='1' model='dmi-to-pci-bridge'/>
    <controller type='pci' index='2' model='pci-bridge'/>
end_build_usb_xml
}

function cal_cmn_build_console_access_xml {
cat << end_build_console_access_xml >> $OUT_XML_FILE
    <serial type='tcp'>
      <source mode='bind' host='$CONSOLE_IPADDR' service='$TTYS0PORT'/>
      <protocol type='telnet'/>
      <target port='0'/>
    </serial>
    <serial type='tcp'>
      <source mode='bind' host='$CONSOLE_IPADDR' service='$TTYS1PORT'/>
      <protocol type='telnet'/>
      <target port='1'/>
    </serial>
end_build_console_access_xml
    cal_cmn_build_more_tcp_xml
}

function cal_cmn_build_more_tcp_xml {
cat << end_build_unused_tcp_xml >> $OUT_XML_FILE
    <console type='tcp'>
      <source mode='bind' host='$CONSOLE_IPADDR' service='$TTYS0PORT'/>
      <protocol type='telnet'/>
      <target type='serial' port='0'/>
    </console>
end_build_unused_tcp_xml
}

function cal_cmn_build_console_access_lxc_xml {
cat << end_build_console_lxc_xml >> $OUT_XML_FILE
    <console type='tcp'>
      <source mode='bind' host='' service='$HVC0PORT'/>
      <protocol type='telnet'/>
      <target port='0'/>
      <alias name='console0'/>
    </console>
    <console type='tcp'>
      <source mode='bind' host='' service='$HVC1PORT'/>
      <protocol type='telnet'/>
      <target port='1'/>
      <alias name='console1'/>
    </console>
end_build_console_lxc_xml
}

function cal_cmn_build_cmdline_start_xml {
cat << end_build_cmdline_start_xml >> $OUT_XML_FILE
  <$VIRT_ARCH:commandline>
end_build_cmdline_start_xml
}

function cal_cmn_build_cmdline_end_xml {
cat << end_build_cmdline_end_xml >> $OUT_XML_FILE
  </$VIRT_ARCH:commandline>
end_build_cmdline_end_xml
}

function cal_cmn_build_arg_xml {
cat << end_build_arg_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='$OPTION'/>
    <$VIRT_ARCH:arg value='$OPTION_ARG'/>
end_build_arg_xml
}

function cal_cmn_build_emulator_xml {
cat << end_build_emulator_xml >> $OUT_XML_FILE
    <emulator>$VIRT_EMULATOR</emulator>
end_build_emulator_xml
}

function cal_cmn_build_common_args_xml {
cat << end_build_char_devices_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='virtio-serial'/>
    <$VIRT_ARCH:arg value='-chardev'/>
    <$VIRT_ARCH:arg value='socket,id=vserial0,host=$CONSOLE_IPADDR,port=$HVC0PORT,telnet,server,nowait'/>
    <$VIRT_ARCH:arg value='-chardev'/>
    <$VIRT_ARCH:arg value='socket,id=vserial1,host=$CONSOLE_IPADDR,port=$HVC1PORT,telnet,server,nowait'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='virtconsole,chardev=vserial0'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='virtconsole,chardev=vserial1'/>
end_build_char_devices_xml
}

# Calculate the offset for the partition
function get_fs_offset {
    local start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $2}')
    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')
    fi
    local offset=$(expr ${start} \* 512)
    echo ${offset}
}

# Calculate the sizelimit for the partition
function get_fs_sizelimit {
    local result
    local sizelimit
    local start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $2}')
    local endlimit=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')

    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')
        endlimit=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $4}')
    fi
    result=$((${endlimit} - ${start}))
    sizelimit=$(expr ${result} \* 512)
    echo ${sizelimit}
}

# Get committed LVID
function get_lvid() {
    #local dev=/dev/sda2
    local dev=""
    local install_repo_path=install_repo/
    local tmp_path=/tmp/
    local offset=0
    local sizelimit=0
 
    declare -F pd_get_install_repo_dev &> /dev/null
    if [ $? -eq 0 ]
    then
        dev=$(pd_get_install_repo_dev)
    else
        dev=/dev/sda2
    fi

    offset=$(get_fs_offset "${dev}" "p1")
    sizelimit=$(get_fs_sizelimit "${dev}" "p1")
 
    mkdir -p ${tmp_path}${install_repo_path}
    mount -o loop ${dev} -o offset=${offset},sizelimit=${sizelimit} ${tmp_path}${install_repo_path}
    lvid=`cat ${tmp_path}${install_repo_path}/local/calvados_lv_state.txt  | cut -d " " -f 1`
    local ignore_stdout=`umount -d ${tmp_path}${install_repo_path}`
    echo ${lvid}
}

#This function needs to be revised once disk detection work is done
function cal_cmn_build_block_devices_xml_internal {

    echo $BOARDTYPE

    # Default boot and data volume
    readonly DEFAULT_BOOT_VOL="${VG_PATH}calvados_lv0"
    readonly DEFAULT_DATA_VOL="${VG_PATH}calvados_data_lv0"

    # Default LV node
    LVNODE="${VG_PATH}calvados_lv0"

    # The logic for pick and choose boot/data partition info file is in
    # install commit (EDCS-1195260) 
    #
    # There could be up to 2 files:
    # F1  /misc/config/calvados_launch_path.txt  (committed boot/data partion)
    # F2  /misc/config/calvados_launch_path_active.txt (active boot/data partition)
    #
    # F1 & F2 is used by install commit.
    # The format of all these files is a single line containing a colon separated
    # string with boot partition and data partition, data partition is optional.
    # <BOOT_PARTITON>[:<DATA_PARTITION>]
    # For instance:
    # /dev/panini_vol_grp/calvados_lv0:/dev/panini_vol_grp/calvados_data_lv0
    # The first token is boot partition (mandatory)
    # The second is the data partition (optional)
    #
    # This script looks for the right file to use in the following order,
    # it only needs to find one file and then use it:
    # {
    #    F2
    #    F1
    # }

    PARTITION_INFO_FILE=""
    NEWLVNODE=""
    LVNODE_DATA=""
    lpath='/dev/panini_vol_grp/calvados_lv'

    if [ -f $F_ACTIVE ]; then
        PARTITION_INFO_FILE=$F_ACTIVE
        log_message "Found Active PARTITION_INFO_FILE $PARTITION_INFO_FILE"
        
        declare -F pd_is_calv_launch_path_recovery_supported &> /dev/null           
        if [ $? -eq 0 ]; then
            pd_is_calv_launch_path_recovery_supported 
            if grep  ${lpath} ${F_ACTIVE}
            then
                log_message "Content of $F_ACTIVE : `cat $F_ACTIVE`"
            else
                #If calvados_launch_path_active file is corrupted
                # then let the system boot from calvados_launch_path file
                log_message "Content of $F_ACTIVE : `cat $F_ACTIVE`"
                log_message "Deleting $F_ACTIVE as its corrupted"
                rm ${F_ACTIVE} 
                cal_launch_boot_path_from_file=`cat $F_COMMITTED | cut -d ":" -f 1`
                lvindex=${cal_launch_boot_path_from_file:$len_lpath}
                lvid=$(get_lvid)
                if [ "$lvindex" != "$lvid" ]; then
                    launch_path="$lpath"$lvid
                    log_message "Correcting $F_COMMITTED with $launch_path"
                    echo $launch_path > $F_COMMITTED
                else 
                    log_message "Content of $F_COMMITTED is good"
                fi
                PARTITION_INFO_FILE=$F_COMMITTED
            fi
        fi
    elif [ -f $F_COMMITTED ]; then
        PARTITION_INFO_FILE=$F_COMMITTED
        log_message "Found Committed PARTITION_INFO_FILE $PARTITION_INFO_FILE"

        declare -F pd_is_calv_launch_path_recovery_supported &> /dev/null           
        if [ $? -eq 0 ]; then
            pd_is_calv_launch_path_recovery_supported 
            if grep  ${lpath} ${F_COMMITTED}
            then
                #calvados_launch_path matches /dev/panini_vol_grp/calvados_lv
                log_message "Content of $F_COMMITTED: `cat $F_COMMITTED`"
                len_lpath=${#lpath}
                cal_launch_boot_path_from_file=`cat $F_COMMITTED | cut -d ":" -f 1`
                lvindex=${cal_launch_boot_path_from_file:$len_lpath}
                lvid=$(get_lvid)
                if [ "$lvindex" != "$lvid" ]; then
                    launch_path="$lpath"$lvid
                    log_message "Correcting $F_COMMITTED with $launch_path"
                    echo $launch_path > $F_COMMITTED
                else 
                    log_message "Content of $F_COMMITTED is good"
                fi
            else
                #calvados_launch_path does not match /dev/panini_vol_grp/calvados_lv
                lvid=$(get_lvid)
                launch_path="$lpath"$lvid
                log_message "Content of $F_COMMITTED: `cat $F_COMMITTED`"
                log_message "Correcting $F_COMMITTED with $launch_path"
                echo $launch_path > $F_COMMITTED
            fi
        fi
    fi

    if [ "$PARTITION_INFO_FILE" != "" -a -f "$PARTITION_INFO_FILE" ]; then
        INFO_LINE=`cat $PARTITION_INFO_FILE`
        log_message "PARTITION_INFO_FILE content $INFO_LINE"
        
        OLDIFS="$IFS"
        IFS=':' read -ra PART_ARRAY <<< "$INFO_LINE"
        IFS="$OLDIFS"
        
        if [ ${#PART_ARRAY[@]} -ge 0 ]; then
            NEWLVNODE="${VG_PATH}`basename ${PART_ARRAY[0]}`"
            log_message "From $PARTITION_INFO_FILE found boot volume: ${NEWLVNODE}"
        fi

        if [ ${#PART_ARRAY[@]} -ge 1 ]; then
            LVNODE_DATA=${PART_ARRAY[1]}
            log_message "From $PARTITION_INFO_FILE found data volume: ${LVNODE_DATA}"
        fi
    else
        NEWLVNODE=$DEFAULT_BOOT_VOL
        LVNODE_DATA=$DEFAULT_DATA_VOL
        log_message "PARTITION_INFO_FILE not found, use default values $NEWLVNODE and $LVNODE_DATA"
    fi

    # this is temporary so that we can use lv0 to launch v2
    if [ -e "${NEWLVNODE}" ]; then
        LVNODE="${NEWLVNODE}"
    fi

    if [ "$LVNODE_DATA" == "" -a -b "${DEFAULT_DATA_VOL}" ]; then
        LVNODE_DATA=$DEFAULT_DATA_VOL
    fi
    log_message "boot partition:${LVNODE}"
    log_message "data partition:${LVNODE_DATA}"

    if [ -e "${LVNODE}" ]; then
        #
        # Check if we have virtio "v" or scsi disks "s". Assume all are of 
        # the same type for now. We need to compress all this to one disk
        # anyway.
        #
        if [ -r /dev/vda ]
        then
            DRIVR_TGT_DEV_PREFIX="vda"
        elif [ -r /dev/nvme0n1 ]; then
            DRIVR_TGT_DEV_PREFIX="nvme0n1p"
        else
            DRIVR_TGT_DEV_PREFIX="sda"
        fi

        DRIVE_SRC_DEV="${LVNODE}" 
        DRIVE_TGT_DEV="vda" 
        ALIAS_NAME="virtio-disk0"
        cal_cmn_build_disk_xml

        PASSTHROUGH_INST_REPO_BLK_DEV='y'
        declare -F platform_pass_host_dev_as_fs &> /dev/null &&  platform_pass_host_dev_as_fs
        
        if [[ $PASSTHROUGH_INST_REPO_BLK_DEV == 'y' ]]
        then
            DRIVE_SRC_DEV="/dev/${DRIVR_TGT_DEV_PREFIX}2"
            ALIAS_NAME="virtio-disk1"
            #need to figure out the difference here
            if [ "$VIRT_ARCH" == "lxc" ]; then
            DRIVE_TGT_DEV="vdb1"
            else
                DRIVE_TGT_DEV="vdb"
            fi

            # If source and traget device is declared by PD in bootstrap file
            # let's pick it from there.

            if grep "DRIVE_SRC_DEV" /etc/init.d/calvados_bootstrap.cfg
            then
                DRIVE_SRC_DEV=`grep "DRIVE_SRC_DEV" /etc/init.d/calvados_bootstrap.cfg | cut -d'=' -f2`
            fi

            if grep "DRIVE_TGT_DEV" /etc/init.d/calvados_bootstrap.cfg
            then
                DRIVE_TGT_DEV=`grep "DRIVE_TGT_DEV" /etc/init.d/calvados_bootstrap.cfg | cut -d'=' -f2`
            fi

            if [ "${PLATFORM}" == "iosxrwbd" -a "${BOARDTYPE}" != "CC" ]; then
                local device=/dev/${MAIN_DISK}
                DRIVE_SRC_DEV=${device}$(parted_partition_number ${device} eXR-REPO 6)
                # don't have to re-evaluate DRIVE_SRC_DEV for DCC nodes as it 
                # will remian static to /dev/hdd2
            fi

            cal_cmn_build_disk_xml
        fi

        if [ -e ${LVNODE_DATA} ]; then 
            DRIVE_SRC_DEV="${LVNODE_DATA}" 
            DRIVE_TGT_DEV="vdc" 
            ALIAS_NAME="virtio-disk2"
            cal_cmn_build_disk_xml
        fi 
    fi

    D1_CVDS_LV1=calvados_1

    # SSD disk - /misc/disk1
    DISK1_LV=${VG_PATH}ssd_disk1_${D1_CVDS_LV1}

    # If DISK1_LV1 has not be created, perhaps it is a removable media (m16)
    # check the PCI path.
    if [ ! -e ${DISK1_LV} ]; then
       DISK1_LV=${PCI_PATH}ssd_disk1_${D1_CVDS_LV1}
    fi
  
    if [ -e ${DISK1_LV} ]; then
        DRIVE_SRC_DEV="$DISK1_LV"; 
        DRIVE_TGT_DEV="vdd"; 
        ALIAS_NAME="virtio-disk3"
        cal_cmn_build_disk_xml
        #cal_cmn_build_disk_simple_xml
    fi

    declare -F pd_check_if_obfl_on_disk &>/dev/null 
    if [ $? -eq 0 ]; then
        obfl_on_disk=$(pd_check_if_obfl_on_disk)
        if [ "x${obfl_on_disk}" == "xyes" ]; then
            DRIVE_SRC_DEV=/dev/"$OBFL_DEV"
            DRIVE_TGT_DEV="vdf"
            ALIAS_NAME="virtio-disk4"
            cal_cmn_build_disk_xml
        fi
    fi

    if [[ $PASSTHROUGH_INST_REPO_BLK_DEV == 'y' ]]
    then
        if [ "${PLATFORM}" == "panini" ] && [[ "$SIMULATION" == 0 ]]; then
            if [ "$BOARDTYPE" == "RP" -o "$BOARDTYPE" == "CC" ]; then
                DRIVE_SRC_DEV="/dev/pci_disk1/install_repo_lv"
                ALIAS_NAME="virtio-disk1"
                #need to figure out the difference here
                DRIVE_TGT_DEV="vde"

                # If source and traget device is declared by PD in bootstrap file
                # let's pick it from there.

                if grep "DRIVE_SRC_DEV" /etc/init.d/calvados_bootstrap.cfg
                then
                    DRIVE_SRC_DEV=`grep "DRIVE_SRC_DEV" /etc/init.d/calvados_bootstrap.cfg | cut -d'=' -f2`
                fi

                if grep "DRIVE_TGT_DEV" /etc/init.d/calvados_bootstrap.cfg
                then
                    DRIVE_TGT_DEV=`grep "DRIVE_TGT_DEV" /etc/init.d/calvados_bootstrap.cfg | cut -d'=' -f2`
                fi

                cal_cmn_build_disk_xml
            fi    
        fi
    fi
}

function cal_cmn_build_block_devices_xml {
    VG_PATH="/dev/panini_vol_grp/"
    PCI_PATH="/dev/pci_disk1/"
    cal_cmn_build_block_devices_xml_internal
}

function cal_cmn_build_lxc_block_devices_xml {
    #For LXC, Soft link is not supported 
    # SYSADMIN_PATH is the mount point for mounting the calvados active
    # root partition. Its base name has the same name as the root
    # partition logical volume. The mount point is also saved in 
    # /var/run/sysadmin_path which later on read by the devlist_get script. 
    VG_PATH="/dev/mapper/panini_vol_grp-"
    PCI_PATH="/dev/mapper/pci_disk1-"
    cal_cmn_build_block_devices_xml_internal
    SYSADMIN_PATH="/lxc_rootfs/`basename $LVNODE`"

    set_rootfs_var SYSADMIN_PATH "$SYSADMIN_PATH"
}

function cal_cmn_build_os_lxc_xml {
cat << end_build_os_lxc_xml >> $OUT_XML_FILE
  <os>
    <type>exe</type>
    <init>/sbin/init</init>
  </os>
end_build_os_lxc_xml
}

function cal_cmn_build_fs_lxc_xml {
cat << end_build_fs_lxc_xml >> $OUT_XML_FILE
    <filesystem type='mount' accessmode='passthrough'>
      <source dir='$SYSADMIN_PATH'/>
      <target dir='/'/>
    </filesystem>
end_build_fs_lxc_xml
    PASSTHROUGH_INST_REPO_BLK_DEV='y'
    declare -F platform_pass_host_dev_as_fs &> /dev/null &&  platform_pass_host_dev_as_fs
    if [[ $PASSTHROUGH_INST_REPO_BLK_DEV == 'n' ]]
    then
        cal_cmn_build_fs_host_dev_lxc_xml $INST_REPO_PATH "/install_repo/"
        cal_cmn_build_fs_host_dev_lxc_xml $INT_PXE_TFTPBOOT_DIR $INT_PXE_TFTPBOOT_DIR
    fi
}

function cal_cmn_build_fs_host_dev_lxc_xml {
cat << end_build_fs_host_dev_lxc_xml >> $OUT_XML_FILE
    <filesystem type='mount' accessmode='passthrough'>
      <source dir='$1'/>
      <target dir='$2'/>
    </filesystem>
end_build_fs_host_dev_lxc_xml
}

function cal_cmn_build_memballoon_lxc_xml {
cat << end_build_memballoon_xml  >> $OUT_XML_FILE
    <memballoon model='virtio'>
        <alias name='balloon0'/>
        <address type='pci' domain='0x0000' bus='0x00' slot='0x09' function='0x0'/>
    </memballoon>
end_build_memballoon_xml
}


# If an argument is passed to this function then it indicates that the 
# linux eobc need not be created.  
# Why this negative logic? - most LXC based platforms want the eobc_br 
# created. The ones with h/w switching can opt-out.

function cal_cmn_build_interface_lxc_xml {

    # By default we create the eobc bridge as long as the
    # platform config file var PLATFORM_HOST_BRIDGE is set
    # and the platform does not opt-out from creating eobc.
    # by passing argument $1  
    if [ $# -eq 0 ]; then
        if [[ "$PLATFORM_HOST_BRIDGE" -eq 1 ]]; then
            local eobc=1
        fi
    fi

    if [[ "$PLATFORM_HOST_BRIDGE" -eq 1 ]]; then
cat << end_build_interface_lxc_xml >> $OUT_XML_FILE
    <interface type='network'>
      <source network='local_br'/>
      <target dev='sysadmin-eth0'/>
      <model type='virtio'/>
    </interface>
end_build_interface_lxc_xml
    fi

    if [ $eobc -eq 1 ]; then
cat << end_build_interface_lxc_xml >> $OUT_XML_FILE
    <interface type='network'>
      <source network='ieobc_br'/>
      <target dev='sysadmin-eth1'/>
      <model type='virtio'/>
    </interface>
end_build_interface_lxc_xml
    fi

    if [[ "$PLATFORM" == "iosxrwbd" ]]; then
        echo Defining container network connectivity for iosxrwbd
        if [ "$BOARDTYPE" == "SW" ]; then

cat << end_build_interface_xml >> $OUT_XML_FILE
        <hostdev mode='capabilities' type='net'><source>
        <interface>knetmlap</interface>
        </source></hostdev>
end_build_interface_xml
        fi

        if [ "$BOARDTYPE" == "CC" ]; then
            echo Defining onie_bmc_br container connectivity
cat << end_build_interface_lxc_xml >> $OUT_XML_FILE
    <interface type='network'>
      <source network='onie_bmc_br'/>
      <target dev='sysadmin-eth2'/>
      <model type='virtio'/>
    </interface>
end_build_interface_lxc_xml
        fi
    fi
}

function cal_cmn_build_device_passthrough_xml {
    source /etc/init.d/devlist_get
    devlist_get sysadmin devlistconf _ _ _ _
    for f in $(cat $devlistconf); do
	if [ -b "$f" ]; then
	    cat <<EOF >>$OUT_XML_FILE
    <hostdev mode='capabilities' type='storage'>
      <source>
        <block>$f</block>
      </source>
    </hostdev>
EOF
	elif [ -c "$f" ]; then
	    cat <<EOF >>$OUT_XML_FILE
    <hostdev mode='capabilities' type='misc'>
      <source>
        <char>$f</char>
      </source>
    </hostdev>
EOF
	fi
    done
}


function cal_cmn_build_sec_xml {
cat << end_build_sec_xml >> $OUT_XML_FILE
  <seclabel type='none'/>
end_build_sec_xml
}

#create the config file in xml format 
# $1 argument - no_eobc_br: If platform does not want the
#               common code to create a s/w bridge for that
#               platform.
function cal_cmn_build_lxc_xml {

    local status=0;

    #build the basic piece for virtualization
    cal_cmn_build_virt_base_cfg_xml 
    cal_cmn_build_os_lxc_xml

    #start device tag
    cal_cmn_build_devices_start_xml 

    # Try various location for libvirt_lxc
    VIRT_EMULATOR="/usr/libexec/libvirt_lxc"
    [ -x "$VIRT_EMULATOR" ] || VIRT_EMULATOR="/usr/lib64/libvirt/libvirt_lxc"
    [ -x "$VIRT_EMULATOR" ] || VIRT_EMULATOR="/usr/lib/libvirt/libvirt_lxc"

    cal_cmn_build_emulator_xml

    #build disk device
    cal_cmn_build_lxc_block_devices_xml

    # Platform specific disk device build
    declare -F pd_build_device_xml &>/dev/null && pd_build_device_xml

    #LVNODE is the active calvados root logical volume which is 
    #set by the cal_cmn_build_lxc_block_devices_xml script function.
    mkdir -p $SYSADMIN_PATH

    partcal_offset=$(find_fs_offset "${LVNODE}" "p1")
    partcal_szlmt=$(find_fs_sizelimit "${LVNODE}" "p1")
    log_message "Creating loop device on ${LVNODE}"
    cal_loop_dev=$(losetup -f --show ${LVNODE} -o ${partcal_offset} --sizelimit ${partcal_szlmt})
    if [ "$?" -eq "0" ];
    then
        log_message "Running  fsck on $cal_loop_dev"
        fsck -fy ${cal_loop_dev} &> /tmp/fsck_cal_loop_dev
        status=$?
        if [ $status -eq 0 ]; then
            log_message "No errors found"
        fi
        if [ $status -eq 1 ]; then
            log_message "Errors were found and corrected"
        fi
        if [ $status -gt 1 ]; then
            log_message "Error encountered"
        fi
        losetup -d ${cal_loop_dev}
    else
        log_message "Loop device creation for  $LVNODE failed"
    fi

    #we expect to get pd_calvados_rootfs_mount_option from init_pd
    mount_option=$([ -f /init_pd ] && \
                grep 'pd_calv_rootfs_mount_option' /init_pd | cut -d "=" -f2)
    if [ x"$mount_option" = x"journal_calv_rootfs" ]; then
        mount $LVNODE $SYSADMIN_PATH \
            -o offset=$partcal_offset,sizelimit=$partcal_szlmt,data=journal
    else
        mount $LVNODE $SYSADMIN_PATH -o offset=$partcal_offset,sizelimit=$partcal_szlmt
    fi

    log_message "Relabel filesystem for $LVNODE"
    chroot $SYSADMIN_PATH /etc/rc.d/init.d/0selinux-init noreboot

    if [ -f /misc/config/devlistdata.txt ]; then
        mkdir -p $SYSADMIN_PATH/misc/config/
        cp /misc/config/devlistdata.txt $SYSADMIN_PATH/misc/config/
    fi

    cal_cmn_build_fs_lxc_xml

    cal_cmn_build_usb_xml
    
    cal_cmn_build_console_access_lxc_xml
    cal_cmn_build_memballoon_lxc_xml

    # Transparently pass $1 argument to function below
    cal_cmn_build_interface_lxc_xml $1

    # Pass through devices in Devlist_calvados.conf
    cal_cmn_build_device_passthrough_xml

    cal_cmn_build_devices_end_xml
    cal_cmn_build_sec_xml
    cal_cmn_build_domain_end_xml
}

function cal_cmn_install_revert_host {
    # Install commit : host partition handling grub.cfg and efi 
    mkdir /tmp/grub_boot_dir
    declare -F pd_get_boot_dev &>/dev/null 
    if [ $? -eq 0 ]; then
        boot_dev=$(pd_get_boot_dev)
    else 
        if [ -n "${EFI_PARTITION}" ]; then
            boot_dev="${EFI_PARTITION}"
        else
            boot_dev="/dev/sda4"
        fi
    fi
    mount "${boot_dev}" /tmp/grub_boot_dir

    if [ -f $GRUB_EFI_BACK ]; then
        grep -q "EFI_GRUB_DIR" /etc/init.d/calvados_bootstrap.cfg
        if [ $? -eq 0 ]; then
            echo $EFI_GRUB_DIR
        else
            EFI_GRUB_DIR="EFI/Cisco/"
        fi

        if [ "${PLATFORM}" == "iosxrwbd" -a "${BOARDTYPE}" != "CC" ]; then
            grep -q "EFI_GRUB_DIR_WBD" /etc/init.d/calvados_bootstrap.cfg
            if [ $? -eq 0 ]; then
                echo $EFI_GRUB_DIR_WBD
            else
                EFI_GRUB_DIR_WBD="grub/"
            fi
            log_message "Found grub.efi for IOSXRWBD, reverting host for next reload"
            cp -a $GRUB_EFI_BACK /tmp/grub_boot_dir/$EFI_GRUB_DIR_WBD/grub.efi
        else
            log_message "Found grub.efi, reverting host for next reload"
            cp -a $GRUB_EFI_BACK /tmp/grub_boot_dir/$EFI_GRUB_DIR/grub.efi
        fi

        if [ -f $GRUB_CFG_BACK ]; then
            log_message "Found grub.cfg, reverting host for next reload"
            cp -a $GRUB_CFG_BACK /tmp/grub_boot_dir/$EFI_GRUB_DIR/grub.cfg
        fi
        
    fi
    # There is no grub.efi in iosxrwbd, so handle it explicitly.
    if [ "${PLATFORM}" == "iosxrwbd" -a "${BOARDTYPE}" != "CC" ]; then
        grep -q "EFI_GRUB_DIR_WBD_CFG" /etc/init.d/calvados_bootstrap.cfg
        if [ $? -eq 0 ]; then
            echo $EFI_GRUB_DIR_WBD_CFG
        else
            EFI_GRUB_DIR_WBD_CFG="grub/"
        fi
        if [ -f $GRUB_CFG_BACK ]; then
            log_message "Found grub.cfg for IOSXRWBD, reverting host for next reload"
            cp -a $GRUB_CFG_BACK /tmp/grub_boot_dir/$EFI_GRUB_DIR_WBD_CFG/grub.cfg
        fi
    fi

    if [ -f $GRUB_MENU_LST_BACK ]; then
        log_message "Found menu.lst, reverting host for next reload"
        cp -a $GRUB_MENU_LST_BACK /tmp/grub_boot_dir/boot/grub/menu.lst
    fi

    if [ -f $BZIMAGE_BACK ]; then
        log_message "Found backup bzImage, reverting it for next reload"
        cp -a $BZIMAGE_BACK /tmp/grub_boot_dir/boot/
    fi

    if [ -f $INITRD_BACK ]; then
        log_message "Found backup initrd, reverting it for next reload"
        cp -a $INITRD_BACK /tmp/grub_boot_dir/boot/
    fi

    if [ -f $NBI_BACK ]; then
        log_message "Found backup nbi, reverting it for next reload"
        cp -a $NBI_BACK /tmp/grub_boot_dir/boot/
        declare -F revert_uboot_host_boot_partition &>/dev/null && revert_uboot_host_boot_partition
    fi

    if [ -f $COMMITTED_MARKER_FILE ]; then
        rm -rf $GRUB_EFI_BACK
        rm -rf $GRUB_CFG_BACK
        rm -rf $GRUB_MENU_LST_BACK
        rm -rf $BZIMAGE_BACK
        rm -rf $INITRD_BACK
        rm -rf $NBI_BACK    
    fi
            
    # Install commit: if active file exists we have used it.
    # Safe to delete here so that we boot next time with commit sw. 
    if [ -f $F_ACTIVE ]; then
        log_message "Found Active partition info $F_ACTIVE , deleting it."
        rm -f $F_ACTIVE
    fi
    umount /tmp/grub_boot_dir
}

function cal_cmn_virsh_start_qemu {
    # start the guest
    declare -F platform_log &>/dev/null && platform_log \
        "$FUNCNAME: Starting virsh QEMU container"

    virsh create $OUT_XML_FILE > /dev/null 2>&1

    if [[ $? == 0 ]]; then
      
        cal_cmn_install_revert_host
        
        # Monitor process 
        CALPID=$(ps ax | grep qemu | grep "name ${VMNAME}" | awk '{print $1}') > /dev/null 2>&1
        if [ ! -z ${CALPID} ]; then
            while true; do
                sleep 5
                if [ ! -e /proc/${CALPID} ]; then
                    break;
                fi
                domstate=`virsh domstate ${VMNAME}`
                if [[ "$domstate" == "in shutdown" ]]; then
                    virsh destroy ${VMNAME}
                    break
                fi
            done
        fi
    else 
        declare -F platform_log &>/dev/null && platform_log \
            "$FUNCNAME: Failed to start guest"
    fi
    sleep 10
}

function cal_cmn_virsh_stop_qemu {
    RUNNING_VM=`virsh list | grep -E running | grep "${VMNAME}" | awk -F" " '{ print $2 }'` > /dev/null 2>&1
    if [ "${RUNNING_VM}" != "" ]; then
        virsh shutdown ${RUNNING_VM} > /dev/null 2>&1
        sleep 2
        virsh destroy ${RUNNING_VM}  > /dev/null 2>&1
    fi
}
function cal_cmn_dev_exists {
    local __device_exist=0
    if [[ -f "$1" ]] ; then
        read line < "$1"
        if [ "$line" == "$2" ]; then
            __device_exist=1
        fi
    fi
    eval $3="'$__device_exist'"
}

. /etc/init.d/functions
function cal_cmn_virsh_start_lxc {
    # start the guest
    declare -F platform_log &>/dev/null && platform_log \
        "$FUNCNAME: Starting virsh LXC container"

    $PLATFORM_LOG_EXEC_CONSOLE virsh -c lxc:/// create $OUT_XML_FILE > /dev/null 2>&1
    if [[ $? == 0 ]]; then

        RUNNING_LXC=`ps ax | grep libvirt_lxc | grep "\-\-name ${VMNAME}" | cut -c 1-6`
        if [ "${RUNNING_LXC}" != "" ]; then
          
            cal_cmn_install_revert_host
            
            # Monitor process 
            CALPID=$(ps ax | grep "name ${VMNAME}" | grep lxc | cut -c 1-6) > /dev/null 2>&1
            while true; do
                ps --pid $CALPID > /dev/null 2>&1
                #  echo pid $?
                if [ "$?" -eq "1" ]; then 
                    break
                fi
                sleep 10
            done
        else
            declare -F platform_log &>/dev/null && platform_log \
                "$FUNCNAME: no running container"
        fi
    else 
        declare -F platform_log &>/dev/null && platform_log \
            "$FUNCNAME: Failed to start container"
    fi
    sleep 10
}

function cal_cmn_virsh_stop_lxc {
    RUNNING_VM=`virsh -c lxc:/// list | grep -E running | grep "${VMNAME}" | awk -F" " '{ print $2 }'` > /dev/null 2>&1
    if [ "${RUNNING_VM}" != "" ]; then
        virsh shutdown ${RUNNING_VM} > /dev/null 2>&1
        for i in {1..10} ; do
           STATE=`virsh domstate ${RUNNING_VM}`
           if [ "$STATE" == "shut off" ]; then
                break
           else
                sleep 1
           fi
        done
        if [ $i -eq 10 ]; then
            virsh destroy ${RUNNING_VM}  > /dev/null 2>&1
        fi
    fi
}

function cal_cmn_virsh_start {
    if [ "${PLATFORM}" == "iosxrwbd" ]; then
       python /opt/cisco/hostos/bin/cardmgr_log.py "Start ${VMNAME}"
    fi

    if [ "$VIRT_ARCH" == "lxc" ]; then
        # This is a UIO device ready check which is done 
        # on per platform basis rather than a common one
        # as all platforms dont need to wait on UIO
        # multicast and hot plug
        cal_pd_lxc_start_device_ready
        cal_cmn_virsh_start_lxc
    else
        cal_cmn_virsh_start_qemu
    fi
}

function cal_cmn_virsh_stop {
    declare -F platform_log &>/dev/null && platform_log \
       "Stop calvados"
    if [ "${PLATFORM}" == "iosxrwbd" ]; then
       python /opt/cisco/hostos/bin/cardmgr_log.py "Stop ${VMNAME}"
    fi

    if [ "$VIRT_ARCH" == "lxc" ]; then
        cal_cmn_virsh_stop_lxc
    else 
        cal_cmn_virsh_stop_qemu
    fi
}

