#!/bin/bash
#
# calvados_launch_zermatt_pd.sh
#
# Copyright (c) 2014-2015, 2017-2018 by Cisco Systems, Inc.
# All rights reserved.

TEMPCHECK=/tmp/tempcheck
if [ ! -f $TEMPCHECK ]; then
    DATE=$(date --iso-8601=ns)
    # Change TEMPLOG to a real file for debugging
    TEMPLOG=/dev/null
    echo "$DATE" >> $TEMPLOG
    echo $$ >>  $TEMPLOG
    echo $0 >> $TEMPLOG

    if [ x"${0##*/}" = x"calvados_launch.sh" ]; then
        echo "$DATE" > $TEMPCHECK

        DEVICE=/dev/sda
        THRESHOLD=120
        MAXREBOOT=5
        MARKER=/etc/micron-over-temp
        echo "Running from calvados_launch" >> $TEMPLOG
        MODEL=$(smartctl -i $DEVICE | grep ^"Device Model" | awk '{ print $3 }')
        VER=$(smartctl -i $DEVICE | grep ^"Firmware Version" | awk '{ print $3 }')
        TEMP=$(smartctl  -x $DEVICE | grep ^"Current Temperature" | awk '{ print $3 }')

        if [[ "$TEMP" == "?" ]]; then
            TEMP=128
        fi

        if [[ "$MODEL" =~ "Micron_M600" && "$VER" == "MC03" && "$TEMP" -gt "$THRESHOLD" ]]; then
            if [[ -f $MARKER && `wc -l < $MARKER` -ge "$MAXREBOOT" ]]; then
                 MSG="$DATE: Micron SSD reports temperature $TEMP. Still above threshold ($THRESHOLD) or unknown after $MAXREBOOT reboot, please request an RMA"
                 echo "$MSG" >> $TEMPLOG
                 echo "$MSG" >> $MARKER
                 echo "<3>$MSG" >> /dev/kmsg
            else
                 MSG="$DATE: Micron SSD reports temperature $TEMP (> threshold of $THRESHOLD or unknown). Rebooting"
                 echo "$MSG" >> $TEMPLOG
                 echo "$MSG" >> $MARKER
                 echo "<0>$MSG" >> /dev/kmsg
                 reboot
                 sleep 60
            fi
        else
            rm -f $MARKER
        fi
        echo "Model $MODEL temp $TEMP" >> $TEMPLOG
    fi
fi


function cal_ucs_build_os_kvm_xml {
cat << end_build_os_kvm_xml >> $OUT_XML_FILE
  <os>
    <type arch='x86_64' machine='pc-q35-1.5'>hvm</type>
    <boot dev='hd'/>
  </os>
end_build_os_kvm_xml
}

function dummyIP_for_qemu {
  #
  # Need to set an IPv4 address on the PF interface
  # before we spawn calvados, otherwise qemu fails to bind the port
  # to which we redirect the console.
  # We put the IP on an out-of-the-way VLAN subint to minimize any
  # potential confusion.
  #
  DUMMY_VLAN=4094 # 0xffe
  DUMMY_ADDR=169.254.147.201/16
  PF=$(ifconfig |grep eth-pf|grep -v '\.'|cut -f1 -d' ')
  if [ ! -z ${PF} ]; then
    if [ ! -f /proc/net/vlan/${PF}.${DUMMY_VLAN} ]; then
      vconfig   add ${PF} ${DUMMY_VLAN}
      ifconfig      ${PF}.${DUMMY_VLAN} ${DUMMY_ADDR} # Note: did not 'up' it.
    fi
  fi
}

UCS_DMIDECODE=/usr/sbin/dmidecode
UCS_LSPCI=/sbin/lspci
UCS_DEVICES_DEVIDS="8086:10e8 1137:0043"
UCS_CTRL_ETH_PF="eth-pf1"
UCS_CTRL_ETH_FILE=/tmp/niantic_devices
UCS_MGMT_INDEX_LINE=3                      # Mgmt devices starting line in UCS_CTR_ETHL_FILE
UCS_DATANIANTIC_ETH_PF="eth-pf2"
UCS_DATANIANTIC_ETH_FILE=/tmp/ucs_dataniantic_devices.txt
UCS_DEVICES_FILE=/tmp/ucs_devices.txt

UCS_CALVADOS_MGMT_INTF_BUS_BASE=39
UCS_CALVADOS_CTRL_INTF_BUS_BASE=40

IXGBE_HLREG0=0x4240
IXGBE_HLREG0_LPBK=0x00008000

IXGBE_FCTRL=0x5080
IXGBE_FCTRL_BAM=0x00000400                  # Broadcast Accept Mode 
IXGBE_FCTRL_SBP=0x00000002                  # Store Bad Packet 
IXGBE_FCTRL_MPE=0x00000100                  # Multicast Promiscuous Ena

IXGBE_AUTOC=0x42A0
IXGBE_AUTOC_LMS_MASK=57344                  # 0x7 << 13
IXGBE_AUTOC_LMS_10G_LINK_NO_AN=8192         # 0x1 << 13
IXGBE_AUTOC_FLU=0x00000001

function enable_niantic_loopback {
    local dev
    local val

    dev="$1"

    val=$(ethtool -s $dev get_addr $IXGBE_HLREG0 2>/dev/null | sed  -ne 's/Value *= *0*x*//ip' | tr '[a-z]' '[A-Z]')
    if [ -z $val ]; then
        echo "Failed to find $dev"
        return 1
    fi
    val="0x$val"
    val=$(printf '0x%X\n'  $((val|IXGBE_HLREG0_LPBK)))
    ethtool -s $dev set_addr $IXGBE_HLREG0 $val >/dev/null 2>&1
    
    val=$(ethtool -s $dev get_addr $IXGBE_FCTRL 2>/dev/null | sed  -ne 's/Value *= *0*x*//ip' | tr '[a-z]' '[A-Z]')
    if [ -z $val ]; then
        echo "Failed to find $dev"
        return 1
    fi
    val="0x$val"
    val=$(printf '0x%X\n'  $((val|IXGBE_FCTRL_BAM|IXGBE_FCTRL_SBP|IXGBE_FCTRL_MPE)))
    ethtool -s $dev set_addr  $IXGBE_FCTRL $val >/dev/null 2>&1

    val=$(ethtool -s $dev get_addr $IXGBE_AUTOC 2>/dev/null | sed  -ne 's/Value *= *0*x*//ip' | tr '[a-z]' '[A-Z]')
    if [ -z $val ]; then
        echo "Failed to find $dev"
        return 1
    fi
    val="0x$val"
    val=$((val&(val^IXGBE_AUTOC_LMS_MASK)))
    val=$(printf '0x%X\n'  $((val|IXGBE_AUTOC_LMS_10G_LINK_NO_AN|IXGBE_AUTOC_FLU)))
    ethtool -s $dev set_addr $IXGBE_AUTOC $val >/dev/null 2>&1

    return $?
}

function disable_niantic_loopback {
    local dev
    local val

    dev="$1"

    val=$(ethtool -s $dev get_addr $IXGBE_HLREG0 2>/dev/null | sed  -ne 's/Value *= *0*x*//ip' | tr '[a-z]' '[A-Z]')
    if [ -z $val ]; then
        echo "Failed to find $dev"
        return 1
    fi
    val="0x$val"
    val=$(printf '0x%X\n'  $((val&(val^IXGBE_HLREG0_LPBK))))
    ethtool -s $dev set_addr $IXGBE_HLREG0 $val >/dev/null 2>&1
    
    val=$(ethtool -s $dev get_addr $IXGBE_AUTOC 2>/dev/null | sed  -ne 's/Value *= *0*x*//ip' | tr '[a-z]' '[A-Z]')
    if [ -z $val ]; then
        echo "Failed to find $dev"
        return 1
    fi
    val="0x$val"
    val=$(printf '0x%X\n'  $((val&(val^IXGBE_AUTOC_FLU))))
    ethtool -s $dev set_addr $IXGBE_AUTOC $val >/dev/null 2>&1
    
    return $?
}

function ucs_is_supported {
    local manu prod

    manu=$($UCS_DMIDECODE -t 2 | grep "Manufactur")
    prod=$($UCS_DMIDECODE -t 2 | grep "Product Name")

    if [ "${manu/Cisco Systems//}" != "${manu}" ]; then
        return 0
    fi
    
    return 1
}

function ucs_get_pcie_slot_from_device {

    local dev
    local fullpath
    local rootdev
    local slot
    local prod

    dev="$1"

    fullpath=$(ls -ld "/sys/bus/pci/devices/0000:${dev}" 2>/dev/null \
        | sed 's/.*-> *\(.*\)/\1/')

    # ../../../devices/pci0000:00/0000:00:05.0/0000:05:00.0/0000:06:08.0/0000:09:00.0/0000:0a:02.0/0000:0b:00.1

    rootdev=$(echo $fullpath | cut -d'/' -f6)
    rootdev=$($UCS_LSPCI -n -s $rootdev 2>/dev/null | cut -d' ' -f3)
 
    # The following is the mapping table for UCS C200 and C210
    # root port 3 (8086:340a)                             -> PCIe Slot 1
    # root port 5 (8086:340c) -> PLX Switch (10b5:8624)
    #                            -> :05.0                 -> PCIe Slot 3
    #                            -> :08.0                 -> PCIe Slot 2
    # root port 7 (8086:340e)                             -> PCIe Slot 5
    # root port 9 (8086:3410)                             -> PCIe Slot 4
    slot=0
    case $rootdev in
        8086:340a) 
            # root port 3
            slot=1
            ;;
        8086:340c)
            # root port 5
            # could be slot 2 or 3
            rootdev=$(echo $fullpath | cut -d'/' -f8)
            if [ "${rootdev/:05.0//}" != "${rootdev}" ]; then
                slot=3
            elif [ "${rootdev/:08.0//}" != "${rootdev}" ]; then
                slot=2
            fi
            ;;
        8086:340e)
            slot=5
            ;;
        8086:3410)
            slot=4
            ;;
    esac
    # for C200, Slot 5 is Slot 2
    prod=$($UCS_DMIDECODE -t 2 | grep "Product Name")
    if [ "${prod/R200-//}" != "${prod}" ]; then
        case $slot in
            1) 
                ;;
            5) 
                slot=2
                ;;
            *)
                slot=0
                ;;
        esac
    fi
    echo $slot
}

function ucs_get_int_giga_mac {
    local idx
    local macs
    local mac

    idx="$1"

    macs=$($UCS_DMIDECODE -t 177 | grep "..:..:..:..:..:..")

    if [ "$idx" == "1" ]; then
        mac=$(echo "$macs" | head -1 2>/dev/null)
    elif [ "$idx" == "2" ]; then
        mac=$(echo "$macs" | tail -1 2>/dev/null)
    else
        mac="UNKNOWN"
    fi
    echo ${mac} | tr '[A-Z]' '[a-z]'
}

function ucs_get_int_giga_bus {
    local idx
    local buses
    local devid='8086:10c9'

    idx="$1"

    buses=$($UCS_LSPCI -d $devid | cut -d' ' -f1)

    #echo "buses = $buses"

    if [ "$idx" == "1" ]; then
        echo "$buses" | head -1 2>/dev/null
    elif [ "$idx" == "2" ]; then
        echo "$buses" | tail -1 2>/dev/null
    else
        echo "UNKNOWN"
    fi
}

# We are using VFs of first Niantic 
# Intel Ether interface for system CtrlEth and MgmtEth.
#
# However, there are 2 such interfaces. As we
# create 4 VFs each, we are getting 8 VFs in total.
# The trouble now is which 4 belong to the 1st Niantic NIC.
# The lspci output is useless in this case.
# We are using the /sys file system for this purpose. 
#
# Usage of the PFs and VFs:
# 
# 1st Niantic Ether:
#     PF: eth-pf1
#     VF1: Calvados Ctrl Ether
#     VF2: XR Ctrl Eth
#     VF3: Calvados MgmtEth
#     VF4: XR MgmtEth
#
# 2nd Niantic Ether:
#     PF: eth-pf2
#     VF1 - VF4: XR Data Eths

function ucs_create_vf_file {

    local pf_name
    local vf_file_name

    pf_name="$1"
    vf_file_name="$2"
    touch $vf_file_name
    #ls -ld /sys/class/net/${pf_name}/device/virtfn* \
    #    | sed 's/.*->.*:\(.*:.*\..*\)/\1/' \
    #    > $vf_file_name 2>/dev/null
}

function ucs_create_ctrl_ether_file {

    # The VF used as Ctrl Ether are from the first 
    # niantic port, which might not have cable connected.
    # In this case, the intf is down and VF cannot
    # communicate to each other. Ctrl Ether will be
    # down in this case. The temp solution is to
    # enable the loopback for the intf for now
    dcbtool sc $UCS_CTRL_ETH_PF dcb off >/dev/null 2>&1 # Need to disable this first
    enable_niantic_loopback $UCS_CTRL_ETH_PF

    # Set up a duppy ip address, otherwise Calvados VM cannot start
    ifconfig $UCS_CTRL_ETH_PF 169.254.13.198

    # Enable the loopback for the second niantic port
    enable_niantic_loopback $UCS_DATANIANTIC_ETH_PF

    # On host, create a file /tmp/niantic_devices with
    # Control Ether device information, for calvados VM.
    # VM manager running in calvados will scp this file
    # into calvados and read all devices. Then assign
    # them to XR VM for the usage of Ctrl Ethernet.
 
    ucs_create_vf_file $UCS_CTRL_ETH_PF $UCS_CTRL_ETH_FILE
}

function ucs_create_dataniantic_ether_file {

    ucs_create_vf_file $UCS_DATANIANTIC_ETH_PF $UCS_DATANIANTIC_ETH_FILE
}

function ucs_get_calvados_mgmt_dev {

    echo $(head -$((UCS_MGMT_INDEX_LINE+0)) $UCS_CTRL_ETH_FILE | tail -1 2> /dev/null)
}

function ucs_get_xr_mgmt_dev {

    echo $(head -$((UCS_MGMT_INDEX_LINE+1)) $UCS_CTRL_ETH_FILE | tail -1 2> /dev/null)
}

function ucs_create_devices_file {

    local data_dev_list="$UCS_DEVICES_DEVIDS"
    local use_2nd_int_giga
    local data_intf_file=$UCS_DEVICES_FILE
    local devid
    local alldev dev 
    
    rm -rf $data_intf_file
    touch $data_intf_file

    # MgmtEther Device for XR
    dev=$(ucs_get_xr_mgmt_dev)
    if [ -n "$dev" ]; then
        echo "M,0,$dev" >> $data_intf_file
    fi

#    # Data Intf 
#    if [ "$1" == "use_2nd_int_giga" ]; then
#        dev=$(ucs_get_int_giga_bus 2)
#        if [ ! -z "$dev" -a "$dev" != "UNKNOWN" ]; then
#            echo "D,0,$dev" >> $data_intf_file
#        fi
#    fi

    # Pass all VFs of second Niantic to XR as Data Intf 
    cat $UCS_DATANIANTIC_ETH_FILE | while read dev ; do
        if [ ! -z "$dev" ]; then
            echo "D,0,$dev" >> $data_intf_file
        fi
    done

    for devid in $data_dev_list; do
        alldev=$($UCS_LSPCI -d $devid | cut -d' ' -f1)
        if [ -z "$alldev" ]; then
            continue
        fi
        for dev in $alldev; do 
            echo "D,$(ucs_get_pcie_slot_from_device $dev),$dev" \
                >> $data_intf_file
        done
    done

    ucs_unbind_data_intf

    return 0
}

function ucs_unbind_device {
    local dev
    local devid

    dev=$1
    
    # Get the vendor and device ID first
    devid=$($UCS_LSPCI -n -s "$dev" 2>/dev/null | cut -d' ' -f 3)
    
    if [ -z "$devid" ]; then
        return
    fi

    devid=${devid/:/ }
    echo "$devid" > /sys/bus/pci/drivers/pci-stub/new_id
    echo "0000:${dev}" > /sys/bus/pci/devices/"0000:${dev}"/driver/unbind
    echo "0000:${dev}" > /sys/bus/pci/drivers/pci-stub/bind
    echo "$devid" > /sys/bus/pci/drivers/pci-stub/remove_id
}

function ucs_unbind_data_intf {

    local data_intf_file=$UCS_DEVICES_FILE
    local dev
    
    while read dev; do 
        if [ -z "$dev" ]; then
            continue
        fi
        ucs_unbind_device ${dev##*,}
    done < $data_intf_file
}

function ucs_get_calvados_dev {

    local dev devlist

    devlist=""

    dev=$(head -1 $UCS_CTRL_ETH_FILE)
    if [ -n "$dev" ]; then
        devlist="${devlist} -device pci-assign,host=${dev},bus=${UCS_CALVADOS_CTRL_INTF_BUS_BASE}"
    fi
    
    dev=$(ucs_get_calvados_mgmt_dev)
    if [ -n "$dev" ]; then
        devlist="${devlist} -device pci-assign,host=${dev},bus=${UCS_CALVADOS_MGMT_INTF_BUS_BASE}"
    fi
    
    echo "$devlist"
}

function ucs_spirit_hostos_run {

    # Create Ctrl Ether file
    ucs_create_ctrl_ether_file

    # Create Data Niantic Ether file
    ucs_create_dataniantic_ether_file

    # Create the data interface device file
    ucs_create_devices_file
}


#build interface
function cal_ucs_build_interface_xml {
cat << end_build_interface_xml >> $OUT_XML_FILE
    <interface type='user'>
      <mac address='52:54:00:29:92:7e'/>
      <model type='virtio'/>
    </interface>
    <interface type='ethernet'>
      <mac address='92:01:01:aa:bb:cc'/>
      <target dev='ce_tap_calv'/>
      <model type='virtio'/>
    </interface>
end_build_interface_xml
}

function cal_ucs_build_udrv_hwid_xml {
    UDRV_HWID_DIR=/dev/udrv_hwid
    if [ ! -d $UDRV_HWID_DIR ]; then
        mkdir -p $UDRV_HWID_DIR
    fi  
cat << end_build_udrv_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='-chardev'/>
    <$VIRT_ARCH:arg value='socket,id=client_$VMNAME,path=$UDRV_HWID_DIR/client_$VMNAME.sock,server,nowait'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='virtserialport,id=vmdev_$VMNAME,chardev=client_$VMNAME,name=udrv_hwid_client'/>
end_build_udrv_xml
}

function calvados_dev_args_ucs {
  echo $(ucs_get_calvados_dev)
}

#setup simluation proxy
function cal_ucs_build_dev_args_xml {
    dev_args=$(calvados_dev_args_ucs) 

    #translate qemu command line to xml 
    OLDIFS="$IFS"
    IFS=' ' read -a array <<< $dev_args
    IFS="$OLDIFS"
    index=0
    while [[ $index -lt ${#array[@]} ]];
    do
        #echo "$index ${array[$index]}"
        OPTION=${array[$index]}
        ((index = index + 1))
        OPTION_ARG=${array[$index]}
        ((index = index + 1))
        echo "$OPTION" 
        echo "$OPTION_ARG"
        cal_cmn_build_arg_xml
    done
}

function cal_ucs_build_misc_xml {
cat << end_build_misc_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='-rtc'/>
    <$VIRT_ARCH:arg value='hostsync=on'/>
    <$VIRT_ARCH:arg value='-readconfig'/>
    <$VIRT_ARCH:arg value='/usr/share/qemu-153/q35-chipset.cfg'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='hwid,debug=1'/>
end_build_misc_xml

}

function cal_ucs_build_connect_redirect_xml {
    OPTION="-redir"
    OPTION_ARG="tcp:50003::22"
    cal_cmn_build_arg_xml
}

#create the config file in xml format 
function cal_ucs_build_xml {
    echo "in calvados_ucs_build_xml"

    cal_cmn_build_virt_base_cfg_xml #build the basic piece for virtualization 
    cal_ucs_build_os_kvm_xml
    
    cal_cmn_build_devices_start_xml #start device tag

    VIRT_EMULATOR="/usr/bin/kvm-qemu-system-x86_64"
    cal_cmn_build_emulator_xml

    cal_cmn_build_block_devices_xml #build disk device

    cal_cmn_build_usb_xml
    CONSOLE_IPADDR="127.0.0.1" #get it from config file
    cal_ucs_build_interface_xml
    cal_cmn_build_console_access_xml
    cal_cmn_build_devices_end_xml

    cal_cmn_build_cmdline_start_xml
    cal_cmn_build_common_args_xml

    cal_ucs_build_udrv_hwid_xml
    cal_ucs_build_dev_args_xml
    cal_ucs_build_misc_xml
    cal_ucs_build_connect_redirect_xml

    cal_cmn_build_cmdline_end_xml
    cal_cmn_build_domain_end_xml
}

# Why this is here ? - This is a pd scecific check where we wait 
#                      on Multicast and Hp for UIO devices to be
#                      ready. All LXC platforms do not support this 
# Description: 
# Start the calvados once the uio dev proxy_hp and muilticast_irq
# have been create if uio are listed on platform, if not then we 
# cannot mknod the uio devices on the calvados/xr.
# 
# Platform Specific need : ZERMATT
# - NOP for Zermatt
function cal_pd_lxc_start_device_ready {
    return 0
}

function calvados_start_pd {
    readonly CALVADOS_COMMON="/opt/cisco/hostos/bin/calvados_launch_common.sh"
    if [ -f $CALVADOS_COMMON ]; then
        . $CALVADOS_COMMON
    fi
 
    if [ "$VIRT_METHOD" == "lxc" ]; then
        VIRT_ARCH="lxc"
    else 
        VIRT_ARCH="qemu"
    fi

    OUT_XML_FILE="/misc/config/${VMNAME}-${VIRT_ARCH}-${PLATFORM}-${SIMULATION}.xml" 
    rm $OUT_XML_FILE

    get_board_type

    if [ "$BOARDTYPE" == "RP" ]; then
        VIRT_SMP=$VIRT_RP_SMP
        VIRT_MEM=$VIRT_RP_MEM
    elif [ "$BOARDTYPE" == "LC" ]; then
        VIRT_SMP=$VIRT_LC_SMP
        VIRT_MEM=$VIRT_LC_MEM
    elif [ "$BOARDTYPE" == "XC" ]; then
        VIRT_SMP=$VIRT_XC_SMP
        VIRT_MEM=$VIRT_XC_MEM
    elif [ "$BOARDTYPE" == "FC" ]; then
        VIRT_SMP=$VIRT_FC_SMP
        VIRT_MEM=$VIRT_FC_MEM
    fi

    ## Board id is derived from [23,28~31] bits in fretta board
    ## after reading register 0x4, so based on that changing the
    ## calvados memory for tortin 16G board.Board id for 16g is
    ## [1,0000 => 10111 => 17]
    local plat_type=$(iofpga_reg_read 0 4)
    num=$( printf "0x%x" $plat_type )
    data=$(($(( num >> 23 & 0x1 )) << 4 | $(( num >> 28 & 0xF ))))
    if [ $data -eq 23 ]
    then
        VIRT_MEM=1048576
        VIRT_RP_MEM=1048576
    fi 

    touch /tmp/niantic_devices

    if [ "$VIRT_ARCH" == "lxc" ]; then
       cal_cmn_build_lxc_xml
    elif (( $SIMULATION )); then
        readonly CALVADOS_SIM="/opt/cisco/hostos/bin/calvados_launch_sim.sh"
        if [ -f $CALVADOS_SIM ]; then
            . $CALVADOS_SIM
        fi
        cal_sim_build_xml
    else
        dummyIP_for_qemu #Niantic only, for now, ncs and ucs
        #create the config file in xml format 
        cal_ucs_build_xml
    fi

    platform_calvados_sdr_xml_check

    # start the guest
    cal_cmn_virsh_start
}

function calvados_stop_pd {
    cal_cmn_virsh_stop
}

