#!/bin/bash
#
# May 2018, Abhishek Ramachandra
#
# Copyright (c) 2010-2018 by cisco Systems, Inc.
# All rights reserved.
#

source /etc/init.d/calvados_bootstrap.cfg
source /opt/cisco/hostos/bin/part_script.sh

cmdline=$(cat /proc/cmdline)
PLATFORM=`echo $cmdline | sed 's/^.*platform=//' | cut -d" " -f1`

function cleanup() {
    [ -d "${PARTMNT}" ] && umount -f ${PARTMNT}
    rmdir ${PARTMNT}
    [ -d "${ISOMNT}" ] && umount -f ${ISOMNT}
    rmdir ${ISOMNT}
    [ -d "${SYSIMGDIR}" ] && rm -rf ${SYSIMGDIR}
}

function clear_dr_efi_parts {
    local dev="/dev/${1}"

    local dr_parts=`parted -s ${dev} print < /dev/null | grep ${PART_REC_NAME} | awk '{print $1}'`
    if [ -n ${dr_parts} ]; then
        echo "${dr_parts}" | xargs -n1 parted -s ${dev} rm > /dev/null 2>&1
    fi
}

function exitclean() {
    dr_log_file "Cleaning up to exit..."
    clear_dr_efi_parts $dr_disk
    cleanup
    exit 1
}

function is_file_compressed {
    file -b $1 | grep -q "compressed"
    return $?
}

function mark_dr_efi_bootable {
    local dev="/dev/${1}"
    PART_NUM_RECOVERY=`parted -s ${dev} print < /dev/null | grep $PART_REC_NAME | awk '{print $1}'`
    parted -s ${dev} set $PART_NUM_RECOVERY boot on
}

function format_dr_efi_partition {
    local dev_name="/dev/${1}${2}"
    local label=${3}

    # Format partition with an FAT16 file system
    dr_log_file "Create vfat filesystem on ${dev_name}"
    
    mkfs.vfat -n ${label} ${dev_name}
    sync
}

function resize_partition_one_fs {
    local dev=/dev/${1}1
    blk_count=`tune2fs -l ${dev} | grep "Block count" | awk '{print $3}'`
    blk_size=`tune2fs -l ${dev} | grep "Block size" | awk '{print $3}'`
    old_size_in_bytes=$(( $blk_count*$blk_size ))
    new_size_in_bytes=$(( $old_size_in_bytes-$DR_EFI_PART_SIZE*1024*1024 ))
    new_blk_count=$(( $new_size_in_bytes/$blk_size ))

    e2fsck -fy ${dev}
    resize2fs ${dev} $new_blk_count
}

function create_dr_efi_partition {
    local dev="/dev/${1}"

    blkid ${dev}
    if [ $? != 0 ]; then
        dr_log_file "Disk ${dev} does not exist!"
    fi

    # resize_partition_one_fs ${1}
    # Implicit assumption of MB as unit size
    part1_end_sector=`parted -s ${dev} print < /dev/null | grep $PART_1_NAME | awk '{print $3}' | tr -dc '0-9'`
    parted -s ${dev} rm 1
    parted -s --align=min ${dev} mkpart $PART_1_NAME 0 $(( $part1_end_sector-$DR_EFI_PART_SIZE ))
    parted -s --align=min ${dev} mkpart $PART_REC_NAME $(( $part1_end_sector-$DR_EFI_PART_SIZE )) $part1_end_sector
}

function extract_grub_files {
    local ISO=${1}
    local grub_dir=${2}
    local sysimg_iso_file=system_image.iso
    local grub_img=/boot/grub2/bootx64.efi
    local grub_cfg=/boot/grub2/grub.cfg

    if [ ! -f $ISO ]; then
        dr_log_file "Cannot locate $ISO. Exiting!"
        exitclean
    fi

    ISOMNT=$(mktemp -d /tmp/ISOMNT.XXXXXX)
    SYSIMGDIR=$(mktemp -d /tmp/ISOMNT.XXXXXX)

    mount -o loop,ro $ISO $ISOMNT || exitclean
    dr_log_file "Mounted recovery image at $ISOMNT"
    
    if is_file_compressed "$ISOMNT/boot/initrd.img"; then
        (zcat $ISOMNT/boot/initrd.img | ( cd $SYSIMGDIR; cpio -idmu --quiet iso/* )) || exitclean
    else
        (cd $SYSIMGDIR; cpio -idmu --quiet iso/* < $ISOMNT/boot/initrd.img ) || exitclean
    fi
    
    if [ -f $SYSIMGDIR/iso/${sysimg_iso_file} ]; then
        isoinfo -R -i $SYSIMGDIR/iso/${sysimg_iso_file} -x "${grub_img}" > ${grub_dir}/bootx64.efi || exitclean
        isoinfo -R -i $SYSIMGDIR/iso/${sysimg_iso_file} -x "${grub_cfg}" > ${grub_dir}/grub.cfg || exitclean
    else
        cp -f ${ISOMNT}/${grub_img} ${grub_dir}
        cp -f ${ISOMNT}/${grub_cfg} ${grub_dir}
    fi
    dr_log_file "Extracted grub files from the recovery image"
}

function install_recovery_efi {
    local dev=/dev/${1}
    local part_num_rec=${2}
    local grub_dir=EFI.prep/boot

    dr_log_file "Install recovery files on EFI ${dev}${part_num_rec}"
    
    PARTMNT=$(mktemp -d /tmp/PARTMNT.XXXXXX)
    mount -o discard ${dev}${part_num_rec} ${PARTMNT} || exitclean
    dr_log_file "Mounted DR EFI partition at ${PARTMNT}"
    mkdir -p ${PARTMNT}/${grub_dir}

    # Check if /misc/disk1 is mounted on the Host
    if [ `mount | grep $HOSTOS_DISK1_PATH | wc -l` -eq 0 ]; then
        dr_log_file "$HOSTOS_DISK1_PATH is not mounted"
        exitclean
    fi
    
    extract_grub_files $HOSTOS_DISK1_PATH/$REC_IMG_PATH/$REC_IMG  ${PARTMNT}/${grub_dir}

    console_speed=115200
    if [ "${PLATFORM}" == "fretta" ]; then
        console_speed=9600
    fi

    sed -i -e "s;speed=115200;speed=${console_speed};" \
       -e "s;root=<P.*>;root=/dev/ram install=/dev/sda recoveryboot;" \
       -e "s;crashkernel=256;crashkernel=192;" \
       -e "s;console=ttyS0,115200;console=ttyS0,${console_speed};" \
       -e "s;Booting from Disk;Booting from Recovery Image;" \
       -e "s;<V.*> root;${HOST_DISK1_LABEL} root\\n \
       loopback loop ${REC_IMG_PATH}/${REC_IMG}\\n \
       set root=(loop);" \
       ${PARTMNT}/${grub_dir}/grub.cfg

    mv ${PARTMNT}/EFI.prep ${PARTMNT}/EFI
    sync
    sync

    dr_log_file "Contents of DR EFI partition grub.cfg:"
    cat ${PARTMNT}/EFI/boot/grub.cfg
    cleanup
}

# Begin main body of script
set_dr_globals
exec >> $DR_LOGFILE
exec 2>&1
set -x

dr_disk=${MAIN_DISK[0]}
check_if_partitioned
if [ $? == "0" ]; then
  clear_dr_efi_parts "${dr_disk}"
  create_dr_efi_partition "${dr_disk}"
fi
mark_dr_efi_bootable    "${dr_disk}"
format_dr_efi_partition "${dr_disk}"   "${PART_NUM_RECOVERY}"  "${RECOVERY_PART_LABEL}"
install_recovery_efi    "${dr_disk}"   "${PART_NUM_RECOVERY}"  "${RECOVERY_PART_LABEL}"
dr_log_file "Created DR EFI partition on /dev/${dr_disk}${PART_NUM_RECOVERY} with label ${RECOVERY_PART_LABEL}"

cleanup
exit 0
# End main body of script
