#!/bin/bash
#
# Implementation of forced reimage chassis operation done from host when
# there is no access to SysAdmin VM.
#
# Copyright (c) 2016 by Cisco Systems, Inc.
# All rights reserved.

# spirit_pd.sh script that has all the IOFPGA access functions
source /etc/init.d/spirit_pd.sh

# Set global variables 
force=0

# Show usage for script
function show_help {
    echo "Usage  force_reimage_chassis.sh [opt]"
    echo "-f     skip prompt (used by scripts)"
}

# Prompt user for confirmation
function ask_yes_or_no() {
    read -p "$1 ([y]es or [N]o): "
    case $(echo $REPLY | tr '[A-Z]' '[a-z]') in
        y|yes) echo "yes" ;;
        *)     echo "no" ;;
    esac
}

RP_CONSOLE_REDIRECT_CTRL_REG_ADDR=0xFC
RP_CRCTRL_BIT_START_READY=0x80000000
RP_CRCTRL_SLOT_FIELD_SHIFT=16
RP_CRCTRL_CMD_SET_BOOT_MODE=0xB0
INTERAL_PXE_BOOT_MODE=1
EXTERNAL_PXE_BOOT_MODE=2
#
# Set LC BIOS Boot Mode
function set_lc_bios_boot_mode() {
    local lc_slot=$(($1 + 1))
    local slot_addr=$(($lc_slot << $RP_CRCTRL_SLOT_FIELD_SHIFT))
    local cmd=$(($RP_CRCTRL_CMD_SET_BOOT_MODE + $INTERAL_PXE_BOOT_MODE))
    local val=$(printf 0x%08X $(($RP_CRCTRL_BIT_START_READY + $slot_addr + $cmd)))

    iofpga_reg_write $RP_CONSOLE_REDIRECT_CTRL_REG_ADDR $val

    sleep 0.5
}

RP2RP_COMMAND_REG_ADDR=0x334
RP2RP_CMD_SET_BOOT_MODE=0x50
RP2RP_PEER_RP_CMD_MASK=0x7F
#
# Set Peer RP BIOS Boot Mode
function set_peer_rp_bios_boot_mode() {
    local cmd=$(($RP2RP_CMD_SET_BOOT_MODE + $INTERAL_PXE_BOOT_MODE))
    local val=$(printf 0x%08X $(($cmd & $RP2RP_PEER_RP_CMD_MASK)))

    iofpga_reg_write $RP2RP_COMMAND_REG_ADDR $val

    sleep 0.5
}

RP_BIOS_BOOT_MODE_REG_ADDR=0x50
#
# Set Local RP BIOS Boot Mode
function set_local_rp_bios_boot_mode() {
    iofpga_reg_write $RP_BIOS_BOOT_MODE_REG_ADDR $EXTERNAL_PXE_BOOT_MODE
}

#
# Takeover Mastership (only work from RP0)
function takeover_mastership() {
    iofpga_reg_write $IOFPGA_RP_ARB_CTRL_REG_ADDR 0x1
    sleep 0.5
}

# Parse command line options first
while getopts "f" opt; do
    case $opt in
        f)
            force=1
            ;;
        \?)
            echo "Invalid option"
            show_help
            exit 1
            ;;
    esac
done

# Get board type that is required to use IOFPGA functions defined in
# spirit_pd.sh script.
get_board_type

# Make sure that we are running from RP
if [ "$BOARDTYPE" != "RP" ]; then
    echo "ERROR: Running from wrong card type of $BOARDTYPE. Reimage operation can only be performed from RP"
    exit 1
fi

# Get the IOFPGA base address
init_iofpga_base_addr

echo "WARNING: this script will attempt to set boot mode for all possible LC slots and peer RP regardless if they exist in chassis or not as the card presence info is not available to this script"

# Prompt the user before continue
if [ "$force" -eq 0 ]; then
    if [ $(ask_yes_or_no "Continue") == "no" ]; then
        echo "Skipped!"
        exit 0
    fi
fi

# Make sure that we are running from RP that has the mastership as reimage
# chassis operaiton require the user of console redirect feature that is
# only available to the RP that has the IOFPGA mastership.
# Note that if we are running from RP0, even if we don't have the mastership
# we can force grab the mastership.
im_active=$(pd_check_rp_im_active)
if [ "$im_active" == "0" ]; then
    if [ $(my_slot_id) -eq 27 ]; then
        echo "WARNING: RP0 does not have mastership, force takeover the mastership from RP1 to perform reimage chassis operation ..."
        takeover_mastership
    else
        echo "ERROR: RP1 does not have mastership, cannot perform reimage chassis operation. Try the same operation on RP0"
        exit 1
    fi
fi

# Now blindly set the boot mode of of all the LCs.
# For now only support 8-slot chassis.
for slot in {0..7}
do
    echo "Configuring LC$slot for internal PXE boot ..."
    set_lc_bios_boot_mode $slot
done

# Then the peer RP
echo "Configuring peer RP for internal PXE boot ..."
set_peer_rp_bios_boot_mode

# And then for the local RP
echo "Configuring local RP for external PXE boot ..."
set_local_rp_bios_boot_mode

# Last thing to do is system reset
RP_RESET_CTRL_REG_ADDR=0x84
RP_RESET_CTRL_BIT_SYS_RESET=0x20
echo "Performing system-wide reset to start reimage operation ..."
sleep 0.5
iofpga_reg_write $RP_RESET_CTRL_REG_ADDR $RP_RESET_CTRL_BIT_SYS_RESET
