#!/bin/bash
#
# calvados/fretta_pkg/boot/scripts/spirit_pd.sh
# This script provides PD variables to other scripts.
# Copyright (c) 2014-2022 by Cisco Systems, Inc.
# All rights reserved.

# source PD-function library
if [ -f /etc/rc.d/init.d/pd-functions ]; then
    source /etc/rc.d/init.d/pd-functions
fi
if [ -f /etc/rc.d/init.d/mod_ins/module-load-functions ]; then
    source /etc/rc.d/init.d/mod_ins/module-load-functions
fi

if [ -f /etc/init.d/get_card_inst.sh ]; then
    source /etc/init.d/get_card_inst.sh
fi

if [ -f /opt/cisco/hostos/bin/part_script.sh ]; then
    source /opt/cisco/hostos/bin/part_script.sh
fi

local_vmtype="$(cat /proc/cmdline|awk -F"vmtype=" '{print $2}'|cut -d' ' -f1)"
if [ ! "$local_vmtype" == "xr-vm" ] && [ -f /etc/init.d/load_kernel_modules.sh ]; then
    source /etc/init.d/load_kernel_modules.sh
fi

BOOTSTRAP_FILE="/etc/init.d/calvados_bootstrap.cfg"
if [ -f $BOOTSTRAP_FILE ]; then
    source $BOOTSTRAP_FILE
fi

function is_simulation_platform()
{
    if [ "$qemu_440chk" == "" ]
    then
        local qemu_440chk=`lspci | grep -i 440FX | wc -l`
    fi

    if [ "$qemu_cpuchk" == "" ]
    then
        local qemu_cpuchk=`cat /proc/cpuinfo | grep QEMU | wc -l`
    fi

    local cmdline=`cat /proc/cmdline`

    if [ "$qemu_440chk" != "0" -a "$qemu_cpuchk" != "0" ]; then
        #
        # QEMU is enabled. Could be VXR or sunstone
        #
        echo "QEMU found, simulation enabled"
        return 1
    else
        #
        # QEMU is NOT enabled.
        #
        return 0
    fi
}

function on_sim()
{
    if [ "${SIMULATION}" == "" ]; 
    then
        #
        # Only log the first time we decide on the simulation setting to avoid
        # noise and repeated looking up of the information we need.
        # 
        is_simulation_platform

        if [[ $? != 0 ]]; then
            readonly SIMULATION=1
        else
            readonly SIMULATION=0
        fi
    fi

    return $SIMULATION
}

#--- Set this PLATFORM_HOST_BRIDGE variable to setup bridge on host
#--- Reset PLATFORM_BRIDGE_EXT_CONNECTIVITY so that the bridge is not
#--- connected to an external interface. Both these variables are
#--- interpreted by nwbridge script.
if [[ $PLATFORM_HOST_BRIDGE = "" ]]; then
    readonly PLATFORM_HOST_BRIDGE=1
fi

if [[ $PLATFORM_BRIDGE_EXT_CONNECTIVITY = "" ]]; then
    readonly PLATFORM_BRIDGE_EXT_CONNECTIVITY=0
fi


function get_board_type {
    get_board_type_fretta
}

function get_card_index {
    init_iofpga_base_addr

    CARDINDEX="$(cat /proc/cmdline|awk -F"card_index=" '{print $2}'|cut -d' ' -f1)"

    if [ -z ${BOARDTYPE} ];then
      # assum RP
      card_value=$(iofpga_reg_read 0x4)
      platform_log_console "card_value is ${card_value}"
      # Check if RP-E(0x01573247) type default case assume to be Keystone RP
      if [ "${card_value}" == "0x01573247" ]; then
          CARDINDEX=27024
      elif [ "${card_value}" == "0x15813136" ]; then
          CARDINDEX=27059
      elif [ "${card_value}" == "0x00986FEB" ]; then 
          CARDINDEX=27065
      elif [ "${card_value}" == "0x90744254" ]; then 
          CARDINDEX=27066
      elif [ "${card_value}" == "0x80744254" ]; then 
          CARDINDEX=27067

      else
          CARDINDEX=21014
      fi
    fi

    if [ -z "${CARDINDEX}" ]; then
        card_value=$(iofpga_reg_read 0x4)
        if [ "${card_value}" == "0x01466283" ]; then
            CARDINDEX=27043
        fi
    fi
}

# Nothing to override in Fretta platform
function override_platform_type {
   echo $1
   return
}
 
#
# FRETTA_BRINGUP_HACK_luislu 
# what is the hack: create a /dev/i2c-1 symbolic link to /dev/i2c-0
# why is the hack : we are using the diag 'idprom' tool that need to
#                   access the CPU EEPROM that is defined as /dev/i2c-1
#                   in diag image, but with XR image we only have one
#                   I2C controller driver installed, so it is getting
#                   created as /dev/i2c-0.
# what is required to remove the hack : we either add the 2nd I2C
#                   controller on OS SDK or modify the diag idprom tool
#                   to open /dev/i2c-0 instead of /dev/i2c-1.
#
# On LC create /dev/i2c-1 as symbolic link to /dev/i2c-0 if is not yet
# created.
function fretta_create_dev_i2c_symlink () {
    if [ "${BOARDTYPE}" == "LC" ]; then
        if [ ! -e ${rootdir}/dev/i2c-1 ]; then
            platform_log "Creating symbolic link for /dev/i2c-1"
            ln -s /dev/i2c-0 /dev/i2c-1
        fi
    fi
}

function init_watchdog() {
    init_iofpga_base_addr
    local wdog_ctrl_reg=$(iofpga_reg_read 0x60)
    local wdog_ctrl_reg_int=$(($((wdog_ctrl_reg+0))&$((0xFF))))
    echo 
    if [ $wdog_ctrl_reg_int -eq $((0xFF)) ]; then
        echo "Re-enabling IOFPGA watchdog with timeout of 90 seconds"
        pd_cfg_watchdog_timeout $1
    else    
        echo "IOFPGA watchdog is already enabled with $(printf '0x%02X\n' $wdog_ctrl_reg_int)"
    fi
}

function check_and_replace_nbi() {
    #
    # Compare the NBI file installed by the hostos_pkg with NBI file served by the tftp
    # They have to be same, if not, replace the file served by tftp server
    # This is required due to the following reason :
    #  Say hostos_pg SMU was installed, with this NBI files at the tftp location will be
    #  replaced by the NBI file provided by the new SMU (post_install_script does it)
    #  Now, if user reloaded the card w/o commit, host will bootup with a backup partition
    #  which does not have the SMU. NBI file at tftp should also be reverted, that too 
    #  asap so that membooted cards do not bootup with the stale NBI.
    # This is the right place to check and replace NBI file
    #
    diff /misc/disk1/tftpboot/sc.nbi /opt/cisco/hostos/bin/ncs5500-sysadmin-image.nbi
    if [ $? -ne 0 ] 
    then
        platform_log_console "Replacing sc.nbi/fm.nbi with /opt/cisco/hostos/bin/ncs5500-sysadmin-image.nbi"
        cp /opt/cisco/hostos/bin/ncs5500-sysadmin-image.nbi /misc/disk1/tftpboot/sc.nbi 
    fi
}

function check_for_nbi_initrd_mismatch () {
    readonly BUILD_INFO_FILE=/etc/build-info.txt
    readonly NBI_INFO_FILE=nbi_info.txt
    readonly TFTP_DIR=/misc/disk1/tftpboot
    readonly NBI_INITRD_ACTIVE=sysadmin-nbi-initrd.img
    readonly NBI_INITRD_COMMITTED=sysadmin-nbi-initrd-committed.img
    
    get_board_type
    if [ "${BOARDTYPE}" == "RP" ]; then
        cur_version_rp=`cat $BUILD_INFO_FILE | grep "XR version" | awk '{print $4}'`
        if [ -z "{cur_version_rp}" ]; then
            platform_log_console "Unable to retrieve XR version from $BUILD_INFO_FILE !!"
            return
        fi
        zcat $TFTP_DIR/$NBI_INITRD_ACTIVE | cpio -idmu $NBI_INFO_FILE
        cur_version_nbi_initrd=`cat $NBI_INFO_FILE | grep Version | awk '{print $4}'`
        if [ -z "{cur_version_nbi_initrd}" ]; then
            platform_log_console "Unable to retrieve NBI initrd version from $NBI_INFO_FILE in $TFTP_DIR/$NBI_INITRD_ACTIVE !!"
            return
        fi

        if [ "${cur_version_rp}" != "${cur_version_nbi_initrd}" ]; then
            # Revert to committed NBI initrd
            platform_log_console "RP XR version $cur_version_rp does NOT match with active NBI initrd version $cur_version_nbi_initrd"
            if [ -f $TFTP_DIR/$NBI_INITRD_COMMITTED ]; then
                platform_log_console "Replacing $NBI_INITRD_ACTIVE with $NBI_INITRD_COMMITTED in $TFTP_DIR"
                rm $TFTP_DIR/$NBI_INITRD_ACTIVE
                mv $TFTP_DIR/$NBI_INITRD_COMMITTED $TFTP_DIR/$NBI_INITRD_ACTIVE
                rm $NBI_INFO_FILE 2>&1 /dev/null
            else
                platform_log_console "Unable to locate $TFTP_DIR/$NBI_INITRD_COMMITTED !!"
            fi
        else
            platform_log_console "RP XR version $cur_version_rp matches with active NBI initrd version $cur_version_nbi_initrd"
        fi
    fi
}

function check_and_replace_lc_efi() {
    #
    # ipxe-lc.efi supports secure boot from 6.1.1 onwards but is it backward compatible ie. 
    # it can handle secure images as well as non-secure images. At every boot, verify that
    # correct efi is available under tftpboot folders so that LCs can boot with secure
    # images, if needed.
    # TODO: Check how grub.efi and grub.cfg get copied to the /misc/disk1/tftpboot/ and 
    #       why these files are needed under /misc/disk1/tftpboot
    #
    diff /misc/disk1/tftpboot/ipxe-lc.efi /etc/rc.d/init.d/ipxe-lc.efi
    if [ $? -ne 0 ] 
    then
        platform_log_console "Replacing /misc/disk1/tftpboot/ipxe-lc.efi with /etc/rc.d/init.d/ipxe-lc.efi"
        cp /etc/rc.d/init.d/ipxe-lc.efi /misc/disk1/tftpboot/ipxe-lc.efi  
    fi

    diff /misc/disk1/tftpboot/ipxe-lc.ipxe.slot26 /etc/rc.d/init.d/ipxe-lc.ipxe.slot26
    if [ $? -ne 0 ] 
    then
        platform_log_console "Replacing /misc/disk1/tftpboot/ipxe-lc.ipxe.slot26 with /etc/rc.d/init.d/ipxe-lc.ipxe.slot26"
        cp /etc/rc.d/init.d/ipxe-lc.ipxe.slot26 /misc/disk1/tftpboot/ipxe-lc.ipxe.slot26  
    fi
    
    diff /misc/disk1/tftpboot/ipxe-lc.ipxe.slot27 /etc/rc.d/init.d/ipxe-lc.ipxe.slot27
    if [ $? -ne 0 ] 
    then
        platform_log_console "Replacing /misc/disk1/tftpboot/ipxe-lc.ipxe.slot27 with /etc/rc.d/init.d/ipxe-lc.ipxe.slot27"
        cp /etc/rc.d/init.d/ipxe-lc.ipxe.slot27 /misc/disk1/tftpboot/ipxe-lc.ipxe.slot27  
    fi
    
    diff /misc/disk1/tftpboot/ipxe-fc.ipxe.slot26 /etc/rc.d/init.d/ipxe-fc.ipxe.slot26
    if [ $? -ne 0 ] 
    then
        platform_log_console "Replacing /misc/disk1/tftpboot/ipxe-fc.ipxe.slot26 with /etc/rc.d/init.d/ipxe-fc.ipxe.slot26"
        cp /etc/rc.d/init.d/ipxe-fc.ipxe.slot26 /misc/disk1/tftpboot/ipxe-fc.ipxe.slot26  
    fi

    diff /misc/disk1/tftpboot/ipxe-fc.ipxe.slot27 /etc/rc.d/init.d/ipxe-fc.ipxe.slot27
    if [ $? -ne 0 ] 
    then
        platform_log_console "Replacing /misc/disk1/tftpboot/ipxe-fc.ipxe.slot27 with /etc/rc.d/init.d/ipxe-fc.ipxe.slot27"
        cp /etc/rc.d/init.d/ipxe-fc.ipxe.slot27 /misc/disk1/tftpboot/ipxe-fc.ipxe.slot27 
    fi
}

function check_and_replace_tftpboot_files() {
    check_and_replace_nbi
    check_for_nbi_initrd_mismatch
    check_and_replace_lc_efi
}

#
# function to move ahci interrupt to last processor core
#
function move_irq_for_ahci() {
    NUM_PROC=$(grep -c ^processor /proc/cpuinfo)
    if [[ $NUM_PROC -le 4 ]]; then
        # Too few processors
        return
    fi

    if [[ $(ls -d /proc/irq/*/ahci 2>/dev/null | wc -l) -eq 0 ]]; then
        # No ahci devices
        return
    fi

    platform_log "======== begining move_irq_for_ahci for ${HOST_NAME}.. ========"

    # Set default affinity to skip last core
    # Assume less than 64 cores
    mask=$(printf %x $(((1 << ($NUM_PROC - 1)) - 1)))
    echo $mask > /proc/irq/default_smp_affinity

    platform_log "======== set mask to default_smp_affinity, $mask ========"

    # All ahci goes to last processor core
    for ahci in /proc/irq/*/ahci; do
        echo $(($NUM_PROC - 1)) >> $(dirname $ahci)/smp_affinity_list
    done

    for irq in /proc/irq/[0-9]*; do
        # On x86-based systems, first 16 irqs are typically handled by 
        # ioapic so there is no real need to move them
        # On current arm-based systems, it won't reach this point since
        # they have at most 4 cores 
        if [ $(basename $irq) -lt 16 ]; then continue; fi
        smp_mask=$((0x`cat $irq/smp_affinity` & 0x$mask))

    platform_log "======== in for loop, smp_mask = $smp_mask ========"

        if [ $smp_mask -ne 0 ]; then
            printf %x $smp_mask >  $irq/smp_affinity
        fi
    done

    platform_log "======== at the end of move_irq_for_ahci for ${HOST_NAME}.. ========"
}

function platform_specific_configuration() {
    local VM_TYPE=$1
    local VIRT_METHOD=$2
    local PLATFORM=$3
    get_board_type

    platform_log "platform specific configuration for $VM_TYPE"

    if [ "$VMTYPE" == "xr-vm" ]; then
        mount -o remount -o size=1M /dev
        if [ "${BOARDTYPE}" == "RP" ]; then
            ifconfig eth-mgmt up
        fi
        if [[ ! -h /harddisk: ]]; then
            platform_log "Fretta:creating missing symlink /harddisk: "
            ln -sf  /misc/disk1/ /harddisk:
        fi
    fi

    if [ "$VM_TYPE" == "hostos" ]; then
        set_proc_sys_net

        # to move ahci interrupt to last processor core
        move_irq_for_ahci
        # Set panic on hung task for recovery (increase timeout to 5min to reduce false alarms)
        platform_log "set hung task time out for $VM_TYPE"
        sysctl -w kernel.hung_task_timeout_secs=300 > /dev/null 2>/dev/null
        sysctl -w kernel.hung_task_panic=1 > /dev/null 2>/dev/null

        if [ "${BOARDTYPE}" == "RP" ]; then
            mount_install_repo_on_host
            check_and_replace_tftpboot_files
        fi
    fi

    if [ "$VM_TYPE" == "sysadmin-vm" ]; then
        # Punch watchdog for first time in sysadmin-vm, this will give enough
        # time for WDMON to take over on the handling of the IOFPGA watchdog
        pd_punch_watchdog

        # Sata_fpd_fc.startup to be removed only for FC's with no SSD
        declare -F pd_is_fc_card_without_SSD &>/dev/null
        if [ $? -eq 0 ] && pd_is_fc_card_without_SSD; then
            if [ -e ${rootdir}/opt/cisco/calvados/etc/startup/sata_fpd_fc.startup ]; then
                rm ${rootdir}/opt/cisco/calvados/etc/startup/sata_fpd_fc.startup
            fi
        fi


        # Let Master RP know that SysAdmin VM has booted
        pd_notify_sysadmin_vm_booted

        fretta_create_dev_i2c_symlink
    fi

    # copy static host file to sysadmin of node
    cp /opt/cisco/hostos/etc/fretta_16_hosts /etc/hosts

    if [ "$VM_TYPE" != "hostos" ]; then
        return
    fi

    # To create symlinks for aug4 fc ipxe intermediate files
    # required at IPXE boot to load system iso
    if [ "${BOARDTYPE}" == "RP" ]; then
        if [ ! -e /www/pages/ipxe-fc.ipxe.slot26 ]; then
            ln -sf /misc/disk1/tftpboot/ipxe-fc.ipxe.slot26 /www/pages/ipxe-fc.ipxe.slot26
        fi

        if [ ! -e /www/pages/ipxe-fc.ipxe.slot27 ]; then
            ln -sf /misc/disk1/tftpboot/ipxe-fc.ipxe.slot27 /www/pages/ipxe-fc.ipxe.slot27
        fi
    fi
    # Punch watchdog before loading kernal modules
    pd_punch_watchdog

    load_all_kernel_module

    # FRETTA_BRINGUP_HACK_luislu 
    # what is the hack: enable watchdog
    # why is the hack : CCTRL KLM disable watchdog after it is loaded.
    #                   This create a windows without watchdog until it is
    #                   re-enabled by WDMON from SysAdmin VM.
    # what is required to remove the hack : new CCTRL KLM without the
    #                   watchdog disabling behavior.
    init_watchdog 90

    # FRETTA_BRINGUP_HACK_luislu 
    # what is the hack: Remove tcf-agent start and kill scripts
    # why is the hack : This software component is not used at all and it
    #                   cause more than 15 seconds of delay in graceful
    #                   shutdown case.
    # what is required to remove the hack : new OS SDK wil this removed.
    rm -f /etc/rc*.d/*tcf-agent

    MY_SLOTID=$(my_slot_id)
    case $MY_SLOTID in
        [1-9]|1[0-9]|20) HOST_NAME="host:0_LC`expr $MY_SLOTID - 1`";;
        2[1-6]) HOST_NAME="host:0_FC`expr $MY_SLOTID - 21`";;
        2[78]) HOST_NAME="host:0_RP`expr $MY_SLOTID - 27`";;
        29|30) HOST_NAME="host:0_SC`expr $MY_SLOTID - 29`";;
        *) echo "Invalid slotid";;
    esac
    hostname "$HOST_NAME"

    baseaddr=0x$(grep MMCONFIG /proc/iomem | awk '{print $1}' | sed -e "s;-.*;;")
    stdbuf -oL /opt/cisco/hostos/bin/linkmonhost_ncs $baseaddr > /var/log/linkmonhost.log 2>&1 &

    echo -n "done platform_specific_configuration for ${HOST_NAME}.."

    return
}



#----Are we on simulation platform?----
qemu_440chk=`lspci |grep -i 440FX |wc -l`
qemu_cpuchk=`cat /proc/cpuinfo |grep QEMU | wc -l`
if [ -z $SIMULATION ]; then
    if [ "$qemu_440chk" != "0" -a "$qemu_cpuchk" != "0" ]; then
        readonly SIMULATION=1
    else
        readonly SIMULATION=0
    fi
fi

#------------------- Install patch files ---------#
if [ -f /etc/init.d/xrnginstall-fretta-pd.sh ]; then
    source /etc/init.d/xrnginstall-fretta-pd.sh
fi

# Fretta platform specific hostOS patching
function  xrnginstall_platform_patch_hostos()
{
    local PARTMNT=$1
    local LVG_NAME=$2
    local HOST_LV_NAME=$3
    local PLATFORM=$4
    local SIM=$5
    local VIRT_METHOD=$6

    platform_log "$FUNCNAME"
    platform_log_console "NCS5500: Patch host"
    xrnginstall_fretta_patch_hostos $*
    platform_log_console "NCS5500: Patch host done"
}

# Fretta platform specific calvados patching
function xrnginstall_platform_patch_calvados()
{
    local PARTMNT=$1
    local PLATFORM=$2
    local SIM=$3
    local VIRT_METHOD=$4

    if (( $SIM )); then
        platform_log "$FUNCNAME"
        platform_log_console "NCS5500 DEVELOPMENT: Patch calvados"
        xrnginstall_fretta_patch_calvados $*
        xrnginstall_fretta_sim_patch_common $*
        platform_log_console "NCS5500 DEVELOPMENT: Patch calvados done"
    else
        platform_log "$FUNCNAME"
        platform_log_console "NCS5500: Patch calvados"
        xrnginstall_fretta_patch_calvados $*
        platform_log_console "NCS5500: Patch calvados done"
    fi
}

# Fretta platform specific XR boot partition patching
function xrnginstall_platform_patch_xr()
{
    local XROOT=$1
    local PLATFORM=$2
    local SIM=$3
    local VIRT_METHOD=$4

    if [ -f /etc/rc.d/init.d/pd-functions ]; then
        source /etc/rc.d/init.d/pd-functions
    fi
    platform_log "Patch XR for Fretta"

    platform_log_exec cat /proc/cmdline

    fretta_patch_xr $XROOT

    xrnginstall_fretta_patch_xr $XROOT $PLATFORM

    platform_log "Patch XR for Fretta done"
}

function init_iofpga_base_addr () {
    init_iofpga_base_addr_fretta
}

#
# Functions for accessing IOFPGA registers
#
function iofpga_reg_write () {
    iofpga_reg_write_fretta $1 $2
}

function iofpga_db_reg_write () {
    iofpga_db_reg_write_fretta $1 $2
}

function iofpga_reg_read () {
    iofpga_reg_read_fretta $1
}

function iofpga_db_reg_read () {
    iofpga_db_reg_read_fretta $1
}

function iofpga_reg_write_bar1 () {
    iofpga_reg_write_fretta_bar1 $1 $2
}

#
# Wrapper functions for accessing IOFPGA registers
# iofpga_reg_read/iofpga_reg_write take one more parameter on fixed platform
#
function iofpga_reg_write_common () {
    iofpga_reg_write $2 $3
}

function iofpga_reg_read_common () {
    iofpga_reg_read $2
}

#
# Only cards running x86 CPU support this, so this is applicable only for
# RPs and LCs.
# RP BIOS_BOOT_MODE register is at address 0x50 and LC is at 0x40.
RP_IOFPGA_BIOS_BOOT_MODE_REG_ADDR=0x50
LC_IOFPGA_BIOS_BOOT_MODE_REG_ADDR=0x40
FC_IOFPGA_BIOS_BOOT_MODE_REG_ADDR=0x40
function pd_reset_bios_boot_mode () {
    init_iofpga_base_addr
    if [ "$BOARDTYPE" == "RP" ]; then
        echo "Resetting RP BIOS_BOOT_MODE register"
        iofpga_reg_write $RP_IOFPGA_BIOS_BOOT_MODE_REG_ADDR 0x0
    elif [ "$BOARDTYPE" == "LC" ]; then
        echo "Resetting LC BIOS_BOOT_MODE register"
        iofpga_reg_write $LC_IOFPGA_BIOS_BOOT_MODE_REG_ADDR 0x0
    elif [ "$BOARDTYPE" == "FC" ]; then
        echo "Resetting FC BIOS_BOOT_MODE register"
        #register for FC is same as LC. no need to create another variable
        iofpga_reg_write $FC_IOFPGA_BIOS_BOOT_MODE_REG_ADDR 0x0
    fi
}

#
# Punch the IOFPGA watchdog
IOFPGA_SYS_WDOG_RESET_REG_ADDR=0x64
IOFPGA_SYS_WDOG_RESET_KICK_WDOG_BIT=0x80000000
function pd_punch_watchdog () {
    init_iofpga_base_addr
    echo "Punching IOFPGA watchdog"
    iofpga_reg_write $IOFPGA_SYS_WDOG_RESET_REG_ADDR $IOFPGA_SYS_WDOG_RESET_KICK_WDOG_BIT
}

#
# Configure the IOFPGA watchdog with a specific stage1 timeout value.
# With Fretta there is only one timeout value that apply to both stage1 and
# stage2 watchdog timeout.
# Input value is in seconds, but when we write into the watchdog control
# register we have to divice it by 2 as each unit represent 2 seconds.
IOFPGA_SYS_WDOG_CTRL_REG_ADDR=0x60
function pd_cfg_watchdog_timeout () {
    local wdog_timeout=$(printf 0x%08X $((($1 + 268) / 2)))
    init_iofpga_base_addr
    iofpga_reg_write $IOFPGA_SYS_WDOG_CTRL_REG_ADDR $wdog_timeout
    echo "`date +%Y-%m-%d:%H:%M:%S` Punching watchdog after setting timeout to  $wdog_timeout"
    declare -F pd_punch_watchdog && pd_punch_watchdog
}

#
# Function to disable watchdog
function pd_disable_watchdog () {
    init_iofpga_base_addr
    iofpga_reg_write $IOFPGA_SYS_WDOG_CTRL_REG_ADDR 0xFF
    echo "IOFPGA watchdog disabled"
}

#
# API to find out if current RP is active
IOFPGA_RP_ARB_CTRL_REG_ADDR=0x330
IOFPGA_IM_ACTIVE_RP_MASK=0x10
function pd_check_rp_im_active () {
    local arb_ctrl=$(iofpga_reg_read $IOFPGA_RP_ARB_CTRL_REG_ADDR)
    local im_active=$(($arb_ctrl & $IOFPGA_IM_ACTIVE_RP_MASK))

    if [ $im_active -eq $(($IOFPGA_IM_ACTIVE_RP_MASK)) ]; then
        echo 1
    else
        echo 0
    fi
}

#
# API to find out which RP slot is active (having the IOFPGA mastership)
IOFPGA_BOARD_STATUS_REG_ADDR=0x304
IOFPGA_RP1_ACTIVE_MASK=0x10000
function pd_get_active_rp_slot () {
    init_iofpga_base_addr

    if [ "$BOARDTYPE" == "RP" ]; then
        local im_active=$(pd_check_rp_im_active)

        if [ "$im_active" == "1" ]; then
            # If I'm the active, return my own slot
            echo "$(my_slot_id)"
        else
            # If I'm not the active one, then return the peer slot number
            if [ $(my_slot_id) -eq 27 ]; then
                echo "28"
            else
                echo "27"
            fi
        fi
    else
        local board_status=$(iofpga_reg_read $IOFPGA_BOARD_STATUS_REG_ADDR)
        local rp1_active=$(($board_status & $IOFPGA_RP1_ACTIVE_MASK))

        if [ $rp1_active -eq $(($IOFPGA_RP1_ACTIVE_MASK)) ]; then
            echo "28"
        else
            echo "27"
        fi
    fi
}

function pd_pxe_update_efi_grub () {
    local input_dir=$1
    local grubcfg=${input_dir}/grub.cfg
    init_iofpga_base_addr

    # Allow PIDs to choose their crashkernel parameters. Eyrie RPs/LCs
    # have seen cases of low memory reservation failure in some cases.
    # Utilize high mem for crashkernel allocation for Eyrie.
    local crashkernel_param="400M@0"

    if [ "${BOARDTYPE}" == "RP" ]; then
        baud_rate=9600
        card_value=$(iofpga_reg_read 0x4)
        platform_log_console "card_value is ${card_value}"
        if [ "${card_value}" == "0x01423741" ]; then
            card_index=21014
        elif [ "${card_value}" == "0x01573247" ]; then
            card_index=27024
        elif [ "${card_value}" == "0x15813136" ]; then
            card_index=27059
        elif [ "${card_value}" == "0x01168153" ]; then
            card_index=21000
        elif [ "${card_value}" == "0x00986FEB" ]; then  #EYRIE_TBD
            card_index=27065
            baud_rate=115200
            crashkernel_param="400M,high"
        else
            card_index=21014
        fi
        # EYRIE_TBD - Change console baudrate to 115200 only for EYRIE_RP
        sed -i -e "s;console=ttyS1,115200;console=ttyS0,${baud_rate} card_index=${card_index} mtdparts=physmap-flash.0:256k(SM_LOG),256k(RR_LOG),256k(KLOG),18m(PLOG),512k(mtdoops) printk.always_kmsg_dump=1 printk.kmsg_dump_only_panic=0 pstore.backend=mtdoops intel_idle.max_cstate=0;"\
	           -e "s;--speed=115200;--speed=${baud_rate};"\
               -e "s;cgroup_disable=memory;;"\
               -e "s;crashkernel=256M@0;crashkernel=${crashkernel_param} slub_debug=-;"\
               -e "s;bigphysarea=10M;bigphysarea=256M;" \
               ${grubcfg}
    elif [ "${BOARDTYPE}" == "LC" ]; then
        card_value=$(iofpga_reg_read 0x4)
        platform_log_console "card_value is ${card_value}"
        pci_inband=0
        case "$card_value" in
            0x0011986A)   
                # Fretta Potenza
                card_index=27000 
                tty_val=ttyS1
                ;;
            0x0011986B) 
                # LC POTENZA 100 TCAM
                card_index=27011
                tty_val=ttyS1
                ;;
            0x0011986C)
                # LC POTENZA 100 Coherent
                card_index=27012
                tty_val=ttyS1
                ;;
            0x01505327)
                # LC POTENZA MACSEC
                card_index=27013
                tty_val=ttyS1
                ;;
            0x01505328)
                # LC POTENZA MACSEC 36X100
                card_index=27018
                tty_val=ttyS1
                ;;
            0x01497417)
                # LC POTENZA 40
                card_index=27010 
                tty_val=ttyS1
                ;;
            0x0011986D)
                # LC POTENZA 40 TCAM
                card_index=27017 
                tty_val=ttyS1
                ;;
            0x01505329)
                # LC POTENZA+100 TCAM
                card_index=27023 
                tty_val=ttyS0
                ;;
            0x0150532C)
                # LC POTENZA+MPA
                card_index=27034 
                tty_val=ttyS0
                ;;    
            0x0150532D)
                # LC POTENZA+MPA TCAM
                card_index=27035 
                tty_val=ttyS0
                ;;
            0x01601000)
                # LC Vigor 400 
                card_index=27044 
                tty_val=ttyS0
                pci_inband=1
                ;;
            0x01601001)
                # LC Vigor 400 TCAM
                card_index=27046
                tty_val=ttyS0
                pci_inband=1
                ;;
            0x01601004)
                # LC Potenza+10C
                # Potenza+10C-AT (27064) and Potenza+10C-A (27060) are 
                # being treated as same even though card_index is different.
                card_index=27060
                tty_val=ttyS0
                ;;
            0x01601005)
                # LC Vigor 100
                card_index=27061
                tty_val=ttyS0
                pci_inband=1
                ;;
            0x01601006)
                # LC Vigor 100 SE
                card_index=27062
                tty_val=ttyS0
                pci_inband=1
                ;;
            0x01601007)
                # LC Vigor DCO
                card_index=27063
                tty_val=ttyS0
                pci_inband=1
                ;;
            0x90744254)         #EYRIE_NON_SE
               # LC Eyrie
                card_index=27066
                tty_val=ttyS0
                pci_inband=1
                crashkernel_param="400M,high"
                ;;
            0x80744254)         #EYRIE_SE
               # LC Eyrie
                card_index=27067
                tty_val=ttyS0
                pci_inband=1
                crashkernel_param="400M,high"
                ;;
            0x01601008)
                # LC Vigor MOD LC
                card_index=27068
                tty_val=ttyS0
                pci_inband=1
                ;;
                *)
                card_index=27000
                tty_val=ttyS1
                ;;
        esac

        sed -i -e "s;cgroup_disable=memory;card_index=${card_index} mtdparts=physmap-flash.0:256k(SM_LOG),256k(RR_LOG),256k(KLOG),18m(PLOG),512k(mtdoops),4m(shmoo) printk.always_kmsg_dump=1 printk.kmsg_dump_only_panic=0 pstore.backend=mtdoops intel_idle.max_cstate=0;"\
               -e "s;ttyS0;${tty_val};"\
               -e "s;crashkernel=256M@0;crashkernel=${crashkernel_param} slub_debug=-;"\
               -e "s;bigphysarea=10M;bigphysarea=256M;" \
               ${grubcfg}
        if [ $pci_inband -eq 1 ]; then
            sed -i -e '/linux / s/$/ swiotlb=0x10000/' ${grubcfg}
            platform_log_console "JCQ: inband pci_inband=${pci_inband}"
        else
            platform_log_console "JCQ: not inband pci_inband=${pci_inband}"
        fi
    # Augutus-4 FC handling
    elif [ "${BOARDTYPE}" == "FC" ]; then
        platform_log_console "Updating grub config for FC with Intel CPU"
        card_value=$(iofpga_reg_read 0x4)
        platform_log_console "card_value is ${card_value}"
        case "$card_value" in
            0x01466283)
                # Fretta Aug-4
                card_index=27043
                tty_val=ttyS0
                ;;
        esac

        # TODO: Aug-4 card brinup hack. noapic=1 to avoid console lockup issue
        sed -i -e "s;cgroup_disable=memory;card_index=${card_index} mtdparts=physmap-flash.0:256k(SM_LOG),256k(RR_LOG),256k(KLOG),18m(PLOG),512k(mtdoops),16m(KTRACES) vmalloc=320M noapic=1 printk.always_kmsg_dump=1 printk.kmsg_dump_only_panic=0 pstore.backend=mtdoops intel_idle.max_cstate=0;"\
                -e "s;ttyS1;${tty_val};"\
                -e "s;quiet;;"\
                -e "s;crashkernel=256M@0;crashkernel=${crashkernel_param} slub_debug=-;"\
                -e "s;bigphysarea=10M;bigphysarea=256M;" \
                ${grubcfg}
    fi
    platform_log_console "NCS5500: Patch grub. card_index=${card_index}"

    efi_loc=`basename ${input_dir}`
    if [ "${efi_loc}" != "boot" ]; then
        mv ${input_dir} ${input_dir}/../boot/
    fi
}


# Create dhcp hook that actually download ISO file
# Here, we want to download file specified in ${new_filename}
# If ${new_filename} is not http, then do not proceed
function fretta_create_dhclient_hook {
cat <<EOF > /etc/dhclient-exit-hooks
#!/bin/bash
if [[ \${new_filename} = *"http"* ]]; then
  url=\${new_filename}
  echo \${url}
  /usr/bin/wget -q \${url} -O $1
elif [ -n "\${new_dhcp_server_identifier}" -a -n "\${new_filename}" ]; then
  echo tftp -v -m binary \${new_dhcp_server_identifier} -c get \${new_filename}
  /usr/bin/tftp -v -m binary \${new_dhcp_server_identifier} -c get \${new_filename} $1
fi
EOF
echo "timeout 20;" >> /etc/dhcp/dhclient.conf
}

# Send DHCP messages with following options
# Fake ourself as iPXE client to let server give us ISO image, not config
function fretta_create_dhclient_conf {
  echo "send vendor-class-identifier \"cisco\";" >> /etc/dhcp/dhclient.conf;
  echo "send user-class \"iPXE\";" >> /etc/dhcp/dhclient.conf;
  SerialNumber=`cat /sys/firmware/dmi/entries/1-0/raw | cut -d '' -f 13`
  echo "send dhcp-client-identifier \"$SerialNumber\";" >> /etc/dhcp/dhclient.conf;
}

function pd_int_pxe_download_iso {
    init_iofpga_base_addr
    platform_log_console "Proceeding to download system_image.iso from primary RP"
    isofile=system_image.iso
    fretta_create_dhclient_hook `pwd`/$isofile
    fretta_create_dhclient_conf
    SLOT=$(iofpga_reg_read 0xc)
    let "y= $SLOT % 0x20"
    platform_log_console "SLOT is $SLOT"
    ifconfig lo 127.0.0.1/16

    card_value=$(iofpga_reg_read 0x4)
    platform_log_console "card_value is ${card_value}"
    if [ "${card_value}" == "0x01573247" -o "${card_value}" == "0x15813136" ]; then
        # 0x01573247  Precisa RP
        # 0x15813136  Precisa-2-RP SUP
        # Precisa RP, RP2-E
        # Niantic ports connected to EOBC switch of SC0, SC1 are
        # 04:00.2 and 04:00.3 These BDF values will not chage.
        # 04:00.2 is connected to SC1 and 04:00.3 is connected to SC0
        # But the n/w interface name may change and hence can't be assumed
        # same interface names are used and mapped always to specific BDF.
        # n/w interface names allocated by Linux kernel is purely
        # based on the device probe and initialization.
        sc0_nic_bdf="04:00\.3"
        sc1_nic_bdf="04:00\.2"
        # find the b:d.f from kern.log and read the interface names
        nic_if_name=`cat /var/log/kern.log | grep $sc0_nic_bdf | grep "NIC Link" | head -1 | sed s/^.*eth*// | cut -d":" -f1`
        if [ -z ${nic_if_name} ]; then
            platform_log_console "NIC $sc0_nic_bdf to SC0 EOBC switch is not detected.."
            SC0="eth5"
        else
            SC0="eth$nic_if_name"
            platform_log_console "$SC0 to SC0 EOBC link is active"

        fi

        nic_if_name=`cat /var/log/kern.log | grep $sc1_nic_bdf | grep "NIC Link" | head -1 | sed s/^.*eth*// | cut -d":" -f1`
        if [ -z ${nic_if_name} ]; then
            platform_log_console "NIC $sc1_nic_bdf to SC1 EOBC switch is not detected.."
            SC1="eth4"
        else
            SC1="eth$nic_if_name"
            platform_log_console "$SC1 to SC1 EOBC link is active"
        fi

        ETHMGMT="eth0"
    elif [ "${card_value}" == "0x00986FEB" ]; then
        SC0="eth1"
        SC1="eth1"
        ETHMGMT="eth0"
    else  
        SC0="eth0"
        SC1="eth1"
        ETHMGMT="eth6"
    fi

    ifconfig $SC0 down hw ether 00:00:01:`printf "%x" $y`:00:00
    ifconfig $SC1 down hw ether 00:00:01:`printf "%x" $y`:00:00
    ifconfig $SC0 up
    ifconfig $SC1 up
    ifconfig $ETHMGMT up
    # Let the interface sync up with remote side
    sleep 5
    rm $isofile
    dhcp_wait=0
    while [ ! -e $isofile ]; do
        for intf in $SC0 $SC1 $ETHMGMT
        do
            platform_log_console "Starting on interface $intf"
            ifconfig $intf up
            dhclient -1 $intf
            if [ -e $isofile ]; then
                platform_log_console "dhclient successful for $intf"
                return
            fi
        done
        dhcp_wait=$((dhcp_wait+1))
        if [ $dhcp_wait -gt 10 ]; then
            echo Unable to acquire ISO file, drop to shell. [exit] to continue. 
            dhcp_wait=0
            /bin/bash
        fi
    done
}

# This is a stop-gap fix to get the ISO from external DHCP server
# so that other cards can boot from this ISO. Later, we can try
# regenerating this ISO internally.

# Check for DHCP server in this order: SC0 link, SC1 link, and ethmgmt
# Interface is not renamed at this point, so use eth6 is external.
# Once DHCP server responses, we will execute hook created by
# fretta_create_dhclient_hook
function pd_pxe_update_pxeboot_iso {
    init_iofpga_base_addr
    if [ "${BOARDTYPE}" == "RP" ]; then
        if [ -f /iso/system_image.iso ]; then
            platform_log_console "Image already downloaded from external source"
            return
        fi

        pd_int_pxe_download_iso

    fi
}


# populates the variable DEVVDB1 & VDB1P1
# $1 should be DEVVDB1
# $2 should be VDB1P1
function xrnginstall_platform_set_vdb
{
    if [[ $# -ne 2 ]]; then
        echo "$FUNCNAME: Wrong argument list"
        return 1
    fi
 
    eval $1="/dev/vdb1"
    eval $2="vdb1p1"
}

# Load the platform specific kernel modules here
function platform_load_kernel_module () {
    return
}

function platform_copy_required_kernel_crash_modules() {
    local dest=$1

    # Copying all required modules
    cp /lib/modules/3*/kernel/drivers/md/dm-mod.ko $dest

    return
}

function platform_load_required_kernel_crash_modules() {

  insmod /lib*/modules/dm-mod.ko
}

#
# platform_nfs_mount_core_dir:
# - This function is designed to be used on the SC, or FC to mount the
#   Active RP directory to it.  The purpose is the save the kernel core
#   to the active RP since the SC, and FC have no storage.
#
function platform_nfs_mount_core_dir
{
  slot_num=$(my_slot_id)
  act_rp_slot=$(pd_get_active_rp_slot)
  $LOGGER "active rp is on slot: ${act_rp_slot}"
  rp_nfs_dir="/misc/scratch/core"

  ifconfig lo 127.0.0.1 netmask 255.255.0.0
  ifconfig eth0 127.1.2.${slot_num} up

  mount -t nfs 127.1.1.${act_rp_slot}:${rp_nfs_dir} ${COREDIR} -o nolock 2>/dev/null
  if [ $? != "0" ] ; then
    $LOGGER "nfs mount of rp failed on eth0"
    ifconfig eth0 down;
    ifconfig eth1 127.1.2.${slot_num} up
    mount -t nfs 127.1.1.${act_rp_slot}:${rp_nfs_dir} ${COREDIR} -o nolock 2>/dev/null
    if [ $? != "0" ] ; then
      $LOGGER "nfs mount of rp failed on eth1"
      if [ "$(type -t pd_notify_host_kdump_mount_failed)" = "function" ]; then
          pd_notify_host_kdump_mount_failed
      fi

      return 1
    fi
  fi
  return 0
}
 
function xrnginstall_platform_modprobe_uio_dma_proxy
{
    local BOARDTYPE=`cat /proc/cmdline | grep boardtype=`
    if [ -n "$BOARDTYPE" ]; then
        BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    fi
 
    if [ "$BOARDTYPE" = "LC" ]; then
        modprobe uio_dma_proxy dmainfo="mvdma0=1M,0 mvdma1=1M,1M \
        mvdma2=1M,2M FD_LC=6M,3M"
    else
        modprobe uio_dma_proxy dmainfo="mvdma0=1M,0 mvdma1=1M,1M \
        mvdma2=1M,2M Fabric_SDK=10M,3M Fabric_FGID=40M,13M FD_RP_FC=45M,53M"
    fi
}

#
# Transmit POST code to SC via one-wire
#
# The POST code is set on bits 0:7 and bit 31 has to set to '1' for HW to
# transmit the post code to SC.
RP_IOFPGA_POST_CODE_REG_ADDR=0x250
LC_IOFPGA_POST_CODE_REG_ADDR=0x210
SC_IOFPGA_POST_CODE_REG_ADDR=0x30C
FC_IOFPGA_POST_CODE_REG_ADDR=0x214
START_TRANSMIT_BIT=0x80000000
function pd_send_post_code_to_sc () {
    local post_code=$(printf 0x%08X $(($START_TRANSMIT_BIT + $1)))
    local event_name=$2
    local reg_addr=0
    init_iofpga_base_addr

    if [ "$BOARDTYPE" == "RP" ]; then
        reg_addr=$RP_IOFPGA_POST_CODE_REG_ADDR
    elif [ "$BOARDTYPE" == "LC" ]; then
        reg_addr=$LC_IOFPGA_POST_CODE_REG_ADDR
    elif [ "$BOARDTYPE" == "XC" ]; then
        reg_addr=$SC_IOFPGA_POST_CODE_REG_ADDR
    elif [ "$BOARDTYPE" == "FC" ]; then
        reg_addr=$FC_IOFPGA_POST_CODE_REG_ADDR
    else
       echo "Unknown board type of $BOARDTYPE"
    fi

    if [ "$reg_addr" != "0" ]; then
        echo "Sending $event_name event"
        iofpga_reg_write $reg_addr $post_code

        # Ack interrupt for sending post code in SC
        if  [ "$BOARDTYPE" == "XC" ]; then
            iofpga_reg_write 0x12C 0x0
        fi
    fi
}

#
# Definition of HOST POST Codes
#
# Definition of all supported POST CODES is in the following header file:
# calvados/dc_common_pkg/drivers/card_mgr/ncs5k/include/cmgr_hal_ncs5k_post_code.h
POST_CODE_HOST_OS_STARTED=0x50
POST_CODE_HOST_IMG_INSTALL_STARTED=0x51
POST_CODE_HOST_IMG_INSTALL_DONE=0x52
POST_CODE_HOST_IMG_INSTALL_RESET=0x53
POST_CODE_HOST_SYSDMIN_VM_STARTED=0x54
POST_CODE_HOST_KERNEL_CORE_DUMP_START=0x56
POST_CODE_HOST_KERNEL_CORE_DUMP_END=0x57
POST_CODE_HOST_KERNEL_CORE_MOUNT_ERR=0x58
POST_CODE_HOST_KERNEL_CORE_CREATE_ERR=0x59
POST_CODE_HOST_MTD_FORMAT_STARTED=0x5A
POST_CODE_HOST_MTD_FORMAT_ENDED=0x5B
POST_CODE_HOST_MTD_FORMAT_FAILED=0x5C
POST_CODE_HOST_MTD_MKVOL_FAILED=0x5D
POST_CODE_HOST_DISK_FATAL_EVENT=0x5E
POST_CODE_HOST_DISK_ERROR_EVENT=0x5F

POST_CODE_SYSDMIN_VM_BOOTED=0xA0

#Mini INITRD POST Codes Section
POST_INITRD_RP_NOT_REACHABLE=0xC0
POST_INITRD_IMAGE_NOT_FOUND=0xC1
POST_INITRD_IMAGE_ZERO_SIZE=0xC2
POST_INITRD_IMAGE_DOWNLD_PARTIAL=0xC3
POST_INITRD_WGET_FAILURE=0xC4
POST_RP_HTTPD_NOT_RUNNING=0xC5
POST_INITRD_INIT_STARTED=0xC6
POST_INITRD_IMAGE_DOWNLD_STARTED=0xC7
POST_INITRD_IMAGE_DOWNLD_COMPLETE=0xC8


function pd_notify_rp_not_reachable () {
    pd_send_post_code_to_sc $POST_INITRD_RP_NOT_REACHABLE "RP(s) not reachable"
}

function pd_notify_initrd_img_not_found () {
    pd_send_post_code_to_sc $POST_INITRD_IMAGE_NOT_FOUND "Initrd image not_found"
}

function pd_notify_initrd_img_zero_size () {
    pd_send_post_code_to_sc $POST_INITRD_IMAGE_ZERO_SIZE "Initrd image size is zero"
}

function pd_notify_initrd_img_download_partial () {
    pd_send_post_code_to_sc $POST_INITRD_IMAGE_DOWNLD_PARTIAL "Initrd image partial download"
}

function pd_notify_initrd_wget_failure () {
    pd_send_post_code_to_sc $POST_INITRD_WGET_FAILURE "Initrd wget failure"
}

function pd_notify_rp_httpd_not_running () {
    pd_send_post_code_to_sc $POST_RP_HTTPD_NOT_RUNNING "HTTPD may not be running on the reachable RP"
}

function pd_notify_initrd_init_started () {
    pd_send_post_code_to_sc $POST_INITRD_INIT_STARTED "Initialization for Initrd wget started"
}

function pd_notify_initrd_image_download_started () {
    pd_send_post_code_to_sc $POST_INITRD_IMAGE_DOWNLD_STARTED "Initrd image download started"
}

function pd_notify_initrd_image_download_complete () {
    pd_send_post_code_to_sc $POST_INITRD_IMAGE_DOWNLD_COMPLETE "Initrd image download completed"
}

#
# IOFPGA bootloader control status register
# When bit 12 is set it indicates that the backup image is booted
IOFPGA_BOOTLDR_CNTRL_STATUS_REG_ADDR=0x68
IOFPGA_BOOTLDR_BACKAUP_ACTIVE_BIT=0x1000

#
# IOFPGA bootloader running version register
IOFPGA_BOOTLDR_RUN_VER_REG_ADDR=0x10

#
# Send HOST_STARTED event by writting the POST code into IOFPGA register.
IOFPGA_REG_RESET_REASON=0x88
function pd_notify_host_started () {
    pd_send_post_code_to_sc $POST_CODE_HOST_OS_STARTED "Host OS Started"

    # Show last reset reason register value
    reset_reason=$(iofpga_reg_read $IOFPGA_REG_RESET_REASON)
    echo "Last Reset Reason =  $(printf 0x%08X $reset_reason)"

    # Show watchdog register value
    wdog_reg=$(iofpga_reg_read 0x60)
    echo "Watchdog Register =  $(printf 0x%08X $wdog_reg)"

    # Show SPI boot timer register value
    spiboot_reg=$(iofpga_reg_read 0x68)
    echo "SPI Boot Timer Register =  $(printf 0x%08X $spiboot_reg)"
}

#
# Send IMG_INSTALL_STARTED event by writting the POST code into IOFPGA
# register.
function pd_notify_img_install_started () {
    pd_send_post_code_to_sc $POST_CODE_HOST_IMG_INSTALL_STARTED "Image Install Started"
}

#
# Send IMG_INSTALL_DONE event by writting the POST code into IOFPGA register.
# Also turn on the status LED in amber color to indicate that image
# installation is completed. This is used in USB installation case to tell
# user that it now can remove the USB for the card to auto-reset to boot the
# USB baked image.
# If this is not a USB installation case, then the board will auto-reset so
# the setting of amber color won't have any meaning to user.
# For memboot cards like SC and FC without the eUSB, we want to bypass the
# setting of amber here as there would not be a auto-reset operation and we
# want to delay the setting of amber until pd_notify_sysadmin_vm_started()
# is called later.
function pd_notify_img_install_done () {
    pd_send_post_code_to_sc $POST_CODE_HOST_IMG_INSTALL_DONE "Image Install Completed"

    # Since on final FC we will have eUSB and only lab boards will not have
    # eUSB, so we will consider that only SC board will memboot, so just
    # bypass the SC board and do this setting for all other boards.
    if [ "$BOARDTYPE" != "XC" ]; then
        pd_set_status_led_to_solid_amber
    fi
}

#
# Send IMG_INSTALL_RESET event by writting the POST code into IOFPGA register.
function pd_notify_img_install_reset () {
    pd_send_post_code_to_sc $POST_CODE_HOST_IMG_INSTALL_RESET "Resetting After Image Installation"
}

#
# Send SYSADMIN_VM_STARTED event by writting the POST code into IOFPGA
# register.
function pd_notify_sysadmin_vm_started () {
    pd_send_post_code_to_sc $POST_CODE_HOST_SYSDMIN_VM_STARTED "SysAdmin VM Started"
    # Show watchdog register value
    wdog_reg=$(iofpga_reg_read 0x60)
    echo "Watchdog Register =  $(printf 0x%08X $wdog_reg)"

    # Turn the status LED to amber to indicate that host is ready
    pd_set_status_led_to_solid_amber

}

#
# Send SYSADMIN_VM_BOOTED event by writting the POST code into IOFPGA
# register.
function pd_notify_sysadmin_vm_booted () {
    pd_send_post_code_to_sc $POST_CODE_SYSDMIN_VM_BOOTED "SysAdmin VM Booted"
}

#
# Send KERNEL_CORE_DUMP_START event by writting the POST code into IOFPGA
# register.
function pd_notify_host_kdump_started () {
    pd_send_post_code_to_sc $POST_CODE_HOST_KERNEL_CORE_DUMP_START "Host kernel core dump started"
}

#
# Send KERNEL_CORE_DUMP_END event by writting the POST code into IOFPGA
# register.
function pd_notify_host_kdump_ended () {
    pd_send_post_code_to_sc $POST_CODE_HOST_KERNEL_CORE_DUMP_END "Host kernel core dump ended"
}

#
# Send KERNEL_CORE_MOUNT_ERR event by writting the POST code into IOFPGA
# register.
function pd_notify_host_kdump_mount_failed () {
    pd_send_post_code_to_sc $POST_CODE_HOST_KERNEL_CORE_MOUNT_ERR "Host kernel core dump filesystem mount failed"
}

#
# Send KERNEL_CORE_CREATE_ERR event by writting the POST code into IOFPGA
# register.
function pd_notify_host_kdump_create_failed () {
    pd_send_post_code_to_sc $POST_CODE_HOST_KERNEL_CORE_CREATE_ERR "Host kernel core dump file creation failed"
}

function pd_notify_host_mtd_format_started () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_FORMAT_STARTED "MTD device format started"
}

function pd_notify_host_mtd_format_ended () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_FORMAT_ENDED "MTD device format completed"
}

function pd_notify_host_mtd_format_failed () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_FORMAT_FAILED "MTD device format failed"
}

function pd_notify_host_mtd_mkvol_failed () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_MKVOL_FAILED "MTD device volume creation failed"
}

function pd_notify_host_disk_partition_format_started () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_FORMAT_STARTED "Hard disk partition format started"
}

function pd_notify_host_disk_partition_format_ended () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_FORMAT_ENDED "Hard disk partition format completed"
}

function pd_notify_host_disk_partition_format_failed () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_FORMAT_FAILED "Hard disk partition format failed"
}

function pd_notify_host_disk_partition_sync_failed () {
    pd_send_post_code_to_sc $POST_CODE_HOST_MTD_MKVOL_FAILED "Data synchronization to memory failed"
}

RP_IOFPGA_PWR_CYCLE_CTRL_REG_ADDR=0x74
IOFPGA_BOARD_CONTROL_REG_ADDR=0x300
#
# Called to perform board reset operation
function pd_board_reset () {
    local reg_addr=0x0
    local reg_val=0x0
    init_iofpga_base_addr

    if [ "$BOARDTYPE" == "RP" ]; then
        reg_addr=$RP_IOFPGA_PWR_CYCLE_CTRL_REG_ADDR
        reg_val=0x1
    else
        reg_addr=$IOFPGA_BOARD_CONTROL_REG_ADDR
        reg_val=0x2
    fi

    if [ "$reg_val" != "0" ]; then
        echo "Resetting board via IOFPGA ..."
        iofpga_reg_write $reg_addr $reg_val
    fi
}

IOFPGA_STATUS_LED_REG_ADDR=0x80
#
# Configure the IOFPGA status LED to turn solid amber color. This is achieved
# by turning on both red and green color at the same time.
function pd_set_status_led_to_solid_amber() {
    local reg_addr=$IOFPGA_STATUS_LED_REG_ADDR
    init_iofpga_base_addr

    local old_val=$(iofpga_reg_read $reg_addr)

    # set STS LED bits to solid amber
    local new_val=$(($old_val & 0xc3ffffff))
    new_val=$(($new_val | 0x14000000)) 

    new_val=$(printf 0x%08X $new_val)
    iofpga_reg_write $reg_addr $new_val
}

#
# Calculate the offset for the partition 
function get_fs_offset {
    local start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $2}')
    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')
    fi
    local offset=$(expr ${start} \* 512)
    echo ${offset}
}

#
# Calculate the sizelimit for the partition 
function get_fs_sizelimit {
    local result
    local sizelimit
    local start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $2}')
    local endlimit=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')

    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')
        endlimit=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $4}')
    fi
    result=$((${endlimit} - ${start}))
    sizelimit=$(expr ${result} \* 512)
    echo ${sizelimit}
}

# CHECK:
#       the code has to remain in sync with Disk-functions
# Check/repair filesystem.
# Args: $1-storage device name, $2-mount point
# Return: (error code returned by fsck)
function pd_check_repair_fs {
    local dev=$1
    local mnt=$2
    local status=0

    platform_log_error "$FUNCNAME $dev $mnt called at $(date)" >&107

    platform_log_error "Checking filesystem in $dev $mnt ..."
    #abort check if partition is already mounted
    local mount=`mount | grep "$dev"`
    if [ ! -z "$mount" ]; then
        platform_log_error "Aborting fs check. Device $dev is mounted."
        # equivalent to fsck error code (8) for operational error
        return 8
    fi

    fsck -fy $dev >&107 2>&1
    status=$?
    if [ $status -eq 0 ]; then
        platform_log_error "No errors found"
    fi
    if [ $status -eq 1 ]; then
        platform_log_error "Errors were found and corrected"
    fi
    if [ $status -gt 1 ]; then
        platform_log_error "Error encountered"
    fi
    /sbin/tune2fs -c 0 -i 0 $dev >&107 2>&1
    return $status
}

# Fix ported from CSCvi96682
function pd_mount_install_repo {
    local loop_dev=$1
    local dir=$2

    # Check if journaling is present. if not, create it
    tune2fs -l ${loop_dev} | grep -q "has_journal"
    if [ $? -ne 0 ]; then
        tune2fs -j ${loop_dev}
    fi

    # Mount the repo
    pd_check_repair_fs ${loop_dev} ${dir}
    mount ${loop_dev} ${dir} >&107 2>&1
    if [ $? -ne 0 ]; then
        platform_log_error "Mounting failed: ${loop_dev} ${dir}"
        return 1
    fi

    # Clean un-mount to finalize the FS
    umount ${dir}

    # Disable journaling to delete Journal entries
    tune2fs -O ^has_journal ${loop_dev} >&107 2>&1

    pd_check_repair_fs ${loop_dev} ${dir}

    # Enable journaling
    tune2fs -j ${loop_dev} >&107 2>&1

    # Mount the repo
    mount ${loop_dev} ${dir} >&107 2>&1
    if [ $? -ne 0 ]; then
        platform_log_error "Mounting failed: ${loop_dev} ${dir}"
        return 1
    fi
    return 0
}
# CHECK:
#       the code has to remain in sync with Disk-functions

#
# pxe_install_memboot_smu.py needs to copy SMU rpms from RP host to XC/FM host
# over http
function mount_install_repo_on_host() {
    local dev=/dev/sda2
    local install_repo_path=install_repo/
    local http_path=/www/pages/

    local offset=$(get_fs_offset "${dev}" "p1")
    local sizelimit=$(get_fs_sizelimit "${dev}" "p1")

    mkdir -p ${http_path}${install_repo_path}

    loop=$(losetup -f --show ${dev} -o ${offset} --sizelimit ${sizelimit})
    if [ $? == 0 ]; then
        echo "Created loop device $loop for $dev disk partition for install_repo"
        pd_mount_install_repo ${loop} ${http_path}${install_repo_path}
        status=$?
        if [ $status -eq 1 ];then
            platform_log_error "pd_mount_install_repo failed for ${loop}"
        fi
        # Setup auto-teardown/autoclear for the loop device by detaching
        losetup -d $loop
    else
        platform_log_error "Failed to setup loop dev for install repo $dev : Install repo mounting failed"
    fi
}

# On Fretta RP we need to mount install_repo on the host so that SMU 
# RPMs can be copied over to host of the memboot cards
# As the install_repo is mounted on the host, insted of mounting it 
# again on the Calvados, mounted install_repo is pass through to lxc
# as filesystem
#
function platform_pass_host_dev_as_fs() {
    PASSTHROUGH_INST_REPO_BLK_DEV='y'
    if [ "$BOARDTYPE" == "RP" ]
    then
        PASSTHROUGH_INST_REPO_BLK_DEV='n'
        INST_REPO_PATH="/www/pages/install_repo"
        INT_PXE_TFTPBOOT_DIR="/misc/disk1/tftpboot"
    fi
}

#
# Array of Fretta chassis slot names index with slot number
fretta_slot_name=(NA 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 NA17 NA18 NA19 NA20 FC0 FC1 FC2 FC3 FC4 FC5 RP0 RP1 SC0 SC1)

#
# Get my nodeid string in internal format (e.g. 0_RP0)
function pd_get_my_int_nodeid_str () {
    echo "0_${fretta_slot_name[$(my_slot_id)]}"
}

#
# Get my nodeid string in external format (e.g. 0/RP0)
function pd_get_my_ext_nodeid_str () {
    echo "0/${fretta_slot_name[$(my_slot_id)]}"
}

#
# Function to do board reset via IOFPGA
function iofpga_auto_warm_reset () {
    # Send out notification athat install will do a local reset
    pd_notify_img_install_reset

    # Resetting the BIOS boot mode
    pd_reset_bios_boot_mode

    echo "Automatic rebooting system via IOFPGA after installation ..."
    echo "BOARDTYPE = $BOARDTYPE...."
    # We need to do WARM RESET via IOFPGA. Only RP has a different
    # register to write
    # Add some delay to allow POST CODE to be sent and message to show
    # up on console before asking IOFPGA to reset
    sleep 2
    if [ "$BOARDTYPE" == "RP" ]; then
        iofpga_reg_write $RP_IOFPGA_PWRCYCLE_CTRL_REG_ADDR 0x1
    else
        iofpga_reg_write $IOFPGA_BOARD_CTRL_REG_ADDR 0x2
    fi
    # Give time to IOFPGA to handle the reset request
    sleep 2
}


#
# Implementation of card reboot after installation of PXE image
# Here we are bypassing the default handling in call_reboot() function in
# PI pxe_install.sh script because that implementation is calling "reboot -f"
# to reload the card and during PXE install, the CCTRL KLM is not loaded
# for us to handle reboot_notification to set IOFPGA register.
# So for this case we need to access the IOFPGA register directly to get
# a WARM RESET behavior so all the power domain are cycled.
RP_IOFPGA_PWRCYCLE_CTRL_REG_ADDR=0x74
IOFPGA_BOARD_CTRL_REG_ADDR=0x300
function pd_pxe_call_reboot () {
    # Skip the reboot if AUTOREBOOT is not enabled
    if [ "$AUTOREBOOT" != "yes" ]; then
        # Resetting the BIOS boot mode
        pd_reset_bios_boot_mode

        # Disable watchdog so we don't reset the board for debugging install
        # issues (with "noautoreboot" configured in grub.cfg file)
        pd_disable_watchdog
        return
    fi

    # Call function to do the reset
    iofpga_auto_warm_reset
}

#
# Usage : pd_get_card_inst card_inst
function pd_get_card_inst () {
    local lcard_inst

    get_board_type

    pd_get_card_inst_from_card_type_fretta $BOARDTYPE lcard_inst

    eval "$1=$lcard_inst"
}

# Enable panic on Oops and panic on softlockup.
# panic on hardlockup is not configurable.
function pd_set_kernel_panic_triggers
{
    echo 1 > /proc/sys/kernel/panic_on_oops
    echo 1 > /proc/sys/kernel/softlockup_panic
}

# Check if Shmoo Partition is needed
function pd_is_shmoo_partition_needed
{
    if [ "$BOARDTYPE" == "LC" ];then
        return 1
    else
        return 0
    fi
}

# Check if HDD Partition is needed
function pd_is_hdd_partition_needed
{
    # Augustus-4 (card_index:27043) does not need HDD partition
    # since it makes use of onboard MTD device
    pd_is_augustus_card && [ "${CARDINDEX}" != 27043 ]
}

function pd_is_fc_card_without_SSD {
    get_card_index
    [ "${CARDINDEX}" -eq 27002 ] || [ "${CARDINDEX}" -eq 27003 ] || ["${CARDINDEX}" -eq 27004 ]
}


#
# RP BIOS_BOOT_MODE register is at address 0x50.
# Find boot mode(ipxe/usb)
# ipxe == 2 and usb == 4
RP_DISK_BOOT_VAR=0x00000000
RP_IPXE_BOOT_VAR=0x00000001
RP_EPXE_BOOT_VAR=0x00000002
RP_USB_BOOT_VAR=0x00000004
function pd_get_boot_mode () {
    if [ "${BOARDTYPE}" == "RP" ]; then
        local boot_dev=""
        local platform=$(cat /proc/cmdline | awk -F "platform=" ' { print $2 }' | cut -d " "  -f1)
        local boot_mode_cmdline="$(cat /proc/cmdline|awk -F"boot_mode=" '{print $2}'|cut -d' ' -f1)"
        local boot_mode=""
        if [ ! -z "$boot_mode_cmdline" ]; then
            boot_mode=$(printf 0x%08X $boot_mode_cmdline)
        else
            init_iofpga_base_addr
            boot_mode=$(iofpga_reg_read $RP_IOFPGA_BIOS_BOOT_MODE_REG_ADDR)

            # We are in the pxe_install.sh context. if user triggered boot from 
            # BIOS prompt then boot_mode will have value as 0 (default initialised)
            # so 0 is treated as DISK_BOOT which is not the case here. 
            # Here boot_mode is overwritten to RP_EPXE_BOOT_VAR/RP_IPXE_BOOT_VAR/  
            # RP_USB_BOOT_VAR based on platform/active or standby node/boot_mode 
            # in cmdline for GISO USB boot
            
            if [ $boot_mode == $RP_DISK_BOOT_VAR ]; then
                if [ "${platform}" == "zermatt" ]; then
                    boot_mode=$RP_EPXE_BOOT_VAR
                else
                    local im_active=$(pd_check_rp_im_active)
                    if [ "$im_active" == "1" ]; then
                        boot_mode=$RP_EPXE_BOOT_VAR
                    else
                        boot_mode=$RP_IPXE_BOOT_VAR
                    fi
                fi
            fi
        fi

        if [ $boot_mode == $RP_EPXE_BOOT_VAR ]; then
            boot_dev="EPXE"
        elif [ $boot_mode == $RP_USB_BOOT_VAR ]; then
            boot_dev="USB"
        else
            boot_dev="IPXE"
        fi
        echo "${boot_dev}"
    fi
}

function pd_int_pxe_download_giso {
    pd_int_pxe_download_iso
}

function pd_lc_int_pxe {
    # Aug-4 FC is intel based board and hence using UEFI install path.
    if [ "${BOARDTYPE}" == "LC" -o "${BOARDTYPE}" == "FC" ]; then
        local MAX_WGET_RETRY=10
        local retry_wget_count=0
        isofile=system_image.iso

        SLOT=$(iofpga_reg_read 0xc)
        my_ip="127.1.2.$(($SLOT % 0x20))"
        platform_log_console "my ip is: $my_ip"

        if [ "${BOARDTYPE}" == "FC" ]; then
            # NIC BDFs defined below are fixed
            # and are connected to SC EOBC always.
            local SC0_BDF="06:00\.0"
            local SC1_BDF="06:00\.1"

            # ixgbe still initializing NIC controllers
            # and NIC are not UP yet.
            # before configuring NIC i/f, wait for 5sec
            sleep 5
            nic_if_name=`cat /var/log/kern.log | grep "$SC0_BDF" | grep "Enabled Features" | head -1 | sed s/^.*eth*// | cut -d":" -f1`
            if [ -z ${nic_if_name} ]; then
                platform_log_console "NIC $SC0_BDF to SC0 EOBC switch is not detected.."
                SC0_nic="eth0"
            else
                SC0_nic="eth$nic_if_name"
                platform_log_console "$SC0_nic link to SC0 is active"
            fi

            nic_if_name=`cat /var/log/kern.log | grep $SC1_BDF | grep "Enabled Features" | head -1 | sed s/^.*eth*// | cut -d":" -f1`
            if [ -z ${nic_if_name} ]; then
                platform_log_console "NIC $SC1_BDF to SC1 EOBC switch is not detected.."
                SC1_nic="eth1"
            else
                SC1_nic="eth$nic_if_name"
                platform_log_console "$SC1_nic link to SC1 is active"
            fi
        fi

        while [ $retry_wget_count -lt $MAX_WGET_RETRY ]
        do
            let retry_wget_count+=1
            if [ $retry_wget_count -ne 1 ]; then
                sleep 2
            fi

            platform_log_console "Trying($retry_wget_count) to download ${isofile} "
            if [ -e $isofile ]; then
                rm $isofile
            fi

            src_rp_slot=$(pd_get_active_rp_slot)
            src_rp_ip="127.1.1.$src_rp_slot"
            platform_log_console "Active rp ip is: $src_rp_ip"

            # fixing the subnet of lo interface, otherwise ping repsonded by itself
            ifconfig lo 127.0.0.1 netmask 255.255.0.0
            if [ "${BOARDTYPE}" == "LC" ]; then
		 ifconfig eth0 hw ether 00:00:01:$(printf '%x' $(($SLOT % 0x20))):00:00 $my_ip netmask 255.255.0.0 up
                ping -c 1 -W 1 $src_rp_ip
                if [ $? -ne 0 ]; then
                    platform_log_console "Ping to ${src_rp_ip} failed, start over !"
                    continue
                fi
            elif [ "${BOARDTYPE}" == "FC" ]; then
                local ping_result="false"
                for nic in $SC0_nic $SC1_nic
                do
		     ifconfig $nic hw ether 00:00:01:$(printf '%x' $(($SLOT % 0x20))):00:00 $my_ip netmask 255.255.0.0 up
                    sleep 1
                    ping -c 1 -W 1 $src_rp_ip
                    if [ $? -ne 0 ]; then
                        platform_log_console "Ping to ${src_rp_ip} using ${nic} failed!"
                        ping_result="false"
                        ifconfig $nic down
                        continue
                    fi
                    ping_result="true"
                    break
                done

                if [ "${ping_result}" == "false" ]; then
                    platform_log_console "Ping to ${src_rp_ip} failed, start over !"
                    continue
                fi
            fi

            # Get free ramfs system memory and check if this internal PXE
            # is happening from system_image.iso created post SU or not.
            # The system_image.iso is same as ncs5500-mini.ISO
            # after SU is performed and hence it consumes more memory
            # compare to ISO image created via external PXE

            img_size=`wget --spider http://$src_rp_ip/$isofile --server-response  -O - 2>&1 | sed -ne '/Content-Length/{s/.*: //;p}'`
            platform_log_console "http://$src_rp_ip server returned $isofile size = $img_size bytes"

            ramfs_free=`df -B 1 | tr -s ' ' | grep tmpfs | grep /$ | cut -d " " -f4`
            platform_log_console "ramfs free memory = $ramfs_free bytes."

            if [[ $ramfs_free -lt $img_size ]]; then
                platform_log_console "ERROR: ramfs free memory $ramfs_free bytes is less than $isofile image size $img_size bytes"
                # if internal PXE is from SU image,
                # 'system_image.iso' image already exist under /iso/ folder
                # on ramfs. If exist don't copy from master RP.
                if [ -f "/iso/$isofile" ]; then
                    mv "/iso/$isofile" "/$isofile"

                    if [ ! -f "/$isofile" ]; then
                        platform_log_console "Failed to move /iso/$isofile image. Creating sym link."
                        ln -sf "/iso/$isofile" "/$isofile"
                        if [ "$?" -eq 0 ]; then
                            platform_log_console "PXE is using System Upgrade ISO image, no need to copy system_imge.iso from master RP."
                            return
                        else
                            platform_log_console "ERROR: Symlink /$isofile to /iso/$isofile failed."
                        fi
                    else
                        platform_log_console "PXE is using System Upgrade ISO image, no need to copy system_imge.iso from master RP."
                        return
                    fi
                fi
            else
                platform_log_console "ramfs free memory $ramfs_free bytes can hold the $isofile image of size $img_size bytes..."
            fi

            wget_cmd="wget -T 60 -c --progress=dot:mega http://$src_rp_ip/$isofile"
            platform_log_console "$wget_cmd"
            $wget_cmd
            ret_wget=$?
            if [ $ret_wget -ne 0 ]; then
                platform_log_console "wget error: $ret_wget, start over"
                continue
            fi

            platform_log_console "Punching watchdog to avoid stage2 timeout due to image download timeout issue"
            declare -F pd_punch_watchdog && pd_punch_watchdog

            if [ -e $isofile ] && [ -s $isofile ] ; then
                size="$(wc -c <"$isofile")"
                platform_log_console "Downloaded $isofile of $size bytes from Active RP($src_rp_ip)"
                return
            else
                platform_log_console "Unable to download $isofile from Active RP($src_rp_ip)"
                continue
            fi
        done
        platform_log_console "Unable to download $isofile from Active RP($src_rp_ip) in $retry_wget_count attempts. Rebooting ..."
        reboot
    fi
}

function pd_handle_swtam {
    mkdir -p /misc/config/swtam
    ln -sf /misc/config/swtam /misc/swtam
    platform_log_console "Softlink created /misc/swtam -> /misc/config/swtam"
}

function pd_dhclient_setup {
if [ -f /etc/dhclient-exit-hooks ]; then
    rm /etc/dhclient-exit-hooks
fi
cat <<EOF > /etc/dhclient-exit-hooks
#!/bin/bash

if [[ \${new_bootfile_name} = *"http"* ]]; then
  url=\${new_bootfile_name}
  echo Downloading file using bootfile-name: \${url}
  /usr/bin/wget -q \${url} -O $1
elif [ -n "\${new_dhcp_server_identifier}" -a -n "\${new_bootfile_name}" ]; then
  echo Downloading file using bootfile-name: \${new_bootfile_name}
  tftp_path=\`echo \${new_bootfile_name} | sed "s/tftp:\/\///g" | sed "s/^[^\/]*\///g"\`
  echo tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path}
  /usr/bin/tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path} $1
elif [[ \${new_filename} = *"http"* ]]; then
  url=\${new_filename}
  echo Downloading file using filename: \${url}
  /usr/bin/wget -q \${url} -O $1
elif [ -n "\${new_dhcp_server_identifier}" -a -n "\${new_filename}" ]; then
  echo Downloading file using filename: \${new_filename}
  tftp_path=\`echo \${new_filename} | sed "s/tftp:\/\///g" | sed "s/^[^\/]*\///g"\`
  echo tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path}
  /usr/bin/tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path} $1
fi
EOF
echo "timeout 10;" >> /etc/dhcp/dhclient.conf
}

function pd_get_serial_number {
    local serial_number=""
    serial_number=`dmidecode -s system-serial-number | grep -v "#"`
    echo $serial_number
}

function pd_get_product_name {
    local product_name=""
    product_name=`dmidecode -s system-product-name | grep -v "#"`
    echo $product_name
}

function pd_dhclient_conf {
  local product_name=$(pd_get_product_name)
  local serial_number=$(pd_get_serial_number)
  platform_log_console "Product name: $product_name"
  platform_log_console "Serial number: $serial_number"
  echo "send vendor-class-identifier \"PXEClient:Arch:00009:UNDI:003010:PID:$product_name\";" >> /etc/dhcp/dhclient.conf;
  echo "send user-class \"iPXE-GISO\";" >> /etc/dhcp/dhclient.conf;
  echo "send dhcp-client-identifier \"$serial_number\";" >> /etc/dhcp/dhclient.conf;
}

function pd_dhclient_conf_append_option () {
  local option=$1
  platform_log "Appending Option = $option"
  # In /etc/dhcp/dhclient.conf there are "request" present for other things like
  # subnet-mask, broadcast-address, etc. To add any additional option we need
  # to prepend with "also" keyword to "request" to make it additonal request
  # otherwise only "request" will override older request parameter 
  echo "also request $option ;" >> /etc/dhcp/dhclient.conf;
}

function pd_get_linkstatus() {
    local intf_name=$1
    local link_status=""

    link_status=`ethtool $intf_name | grep "Link detected" | cut -d ":" -f2`
    echo $link_status
}

function pd_copy_from_usb {
    local usbpath=""
    local usbmnt=""
    local removable_drives=""
    local usbgrub=""
    local usbiso=""
    local isofile=$1

    for _device in /sys/block/*/device; do
        if echo $(readlink -f "$_device")|egrep -q "usb"; then
            _disk=$(echo "$_device" | cut -f4 -d/)
            removable_drives="$_disk"
        fi
    done

    usbpath=`blkid -c /dev/null | grep $removable_drives | cut -d ":" -f1`
    platform_log_console "USB path: $usbpath"
    usbmnt=$(mktemp -d /tmp/usbdev.XXXXXX)
    mount $usbpath $usbmnt >&107 2>&1

    if [ -f "$usbmnt/EFI/boot/grub.cfg" ]; then
        usbgrub=$usbmnt/EFI/boot/grub.cfg
    elif [ -f "$usbmnt/boot/grub.cfg" ]; then
        usbgrub=$usbmnt/boot/grub.cfg
    fi

    if [ -n "$usbgrub" ]; then
        echo "Found grub file $usbgrub" >&107
        usbiso=$(grep loopback $usbgrub | head -1 | awk '{print $3}' | sed 's/\r$//')
        echo "Looking for $usbiso on $usbmnt" >&107
        if [ -f "$usbmnt$usbiso" ]; then
            usbiso=$usbmnt$usbiso
            echo "Found iso image at $usbiso" >&107
        else
           usbiso=
           echo "Cannot find iso image from grub.cfg" >&107
        fi
    fi

    if [ -z "$usbiso" ]; then
        if [ -f "$(echo $usbmnt/boot/*.iso*)" ]; then
          usbiso=$(echo $usbmnt/boot/*.iso*)
          echo "Found iso image at $usbiso" >&107
        elif [ -f "$(echo $usbmnt/*.iso*)" ]; then
          usbiso=$(echo $usbmnt/*.iso*)
          echo "Found iso image at $usbiso" >&107
        fi
    fi

    if [ -n "$usbiso" -a -f "$usbiso" ]; then
        cp $usbiso $isofile
    fi

    umount $usbmnt
    rmdir $usbmnt
}


function pd_download_giso {
    local boot_dev=""
    local system_image="system_image.iso"
    local max_dhclient_try=2
    local link_status=""

    mount -r -o loop /iso/$HOST_ISO ${ISOMNT} || exitclean
    local GIsoType=$(check_iso_type ${ISOMNT}) 
    umount ${ISOMNT} || exitclean
    platform_log_console "Golden ISO type = $GIsoType and PKG_VER = $PKG_VER"

    if [[ ${GIsoType} < ${PKG_VER} ]]; then
        platform_log_console "Detected non eXR platform. Exiting pd_download_giso"
        return
    fi

    if [ -f /iso/$system_image ]; then
        mount -r -o loop /iso/$system_image ${ISOMNT} || exitclean
        if [ -f ${ISOMNT}/giso_info.txt ]; then
            platform_log_console "Additional RPMS embedded in $system_image"
            cp /iso/${system_image} /${system_image}
            if (( $? != 0 )); then
                platform_log_console "Failed to copy $system_image"
            fi
            return
        fi   
    fi

    # fretta pd function
    declare -f pd_get_boot_mode >&107 2>&1                                       
    if [ "$?" -eq 0 ]; then                                                      
        boot_dev=$(pd_get_boot_mode)                                             
        if [ -z "${boot_dev}" ]; then                                            
            platform_log_console "Couldn't find boot id(iPXE/USB). Returning from pd_download_giso"
            return                                                               
        fi                                                                       
    fi 

    if [ ! -f /${system_image} ]; then
        # fretta standby rp 
        if [ "${boot_dev}" == "IPXE" ]; then                                      
            platform_log "Current Boot: Internal-iPXE"
            declare -F pd_int_pxe_download_iso >&107 2>&1 && \                
                pd_int_pxe_download_iso ./                                    
        elif [ "${boot_dev}" == "EPXE" ]; then                     
            platform_log "Current Boot: External-iPXE"
            pd_dhclient_setup /${system_image}
            pd_dhclient_conf
            pd_dhclient_conf_append_option "bootfile-name"

            local eths=( $(ifconfig -a | grep "Link\ " | cut -d' ' -f1 ) )
            for eth in "${eths[@]}";
            do
                ifconfig ${eth} up
            done
            sleep 5
            # Fetch ISO from PXE server.
            # Try on each interface.
            local dhc_try=0
            while [ ! \( -f /${system_image} -a `cat /${system_image} | wc -c` -gt 0 \) ]; do
                for eth in "${eths[@]}"
                do
                    link_status=$(pd_get_linkstatus ${eth})
                    if [ "$link_status" == "yes" ]; then
                        platform_log_console "Running dhclient over ${eth}"      
                        dhclient -1 ${eth}
                        if [ -f /${system_image} -a `cat /${system_image} | wc -c` -gt 0 ]; then
                            break 
                        fi
                    fi
                done

                dhc_try=$((dhc_try+1))
                if [ ${dhc_try} -gt ${max_dhclient_try} ]; then
                    platform_log_console "Failed to fetch ISO from DHCP server" 
                    break
                fi
            done
            # tftp download takes huge time for downlaoding the image
            # So to avoid timeout before completion of baking, 
            # watchdog time is increased by punching watchdog
            platform_log_console "Punching watchdog, downloading of whole ISO might time out" 
            declare -F pd_punch_watchdog && pd_punch_watchdog
        elif [ "${boot_dev}" == "USB" ]; then                     
            platform_log_console "Current Boot: USB"
            pd_copy_from_usb /${system_image}
        fi
    fi
}

#
# debug partition is needed
#
function pd_check_if_debug_part_needed {
    return 0
}

# Copy Recovery image from Host /misc/disk1
function pd_get_recovery_image {
    local boardtype="${1}"
    local isofile=system_image.iso
    local rec_img_path="/misc/disk1/recovery"

    # On Fretta, Host SSD Vol is already mounted on /misc/disk1
    if [ ! -f $rec_img_path/$isofile ]; then
       platform_log_console "Unable to locate recovery image at $rec_img_path"
       ( cd $rec_img_path && ls -laR . ) >&107
       return 1
    fi

    platform_log_console "Located recovery image at $rec_img_path/$isofile"
    cp $rec_img_path/$isofile $isofile
    chmod 544 $isofile
}

# Backup Disaster Recovery partition during bake phase if it exists
function pd_check_for_dr_efi_part {
    check_if_partitioned >&107 2>&1
    if [ $? == 1 ]; then 
        DR_EFI_EXISTS=true
        echo "Disaster Recovery partition exists already" >&107
    else
        DR_EFI_EXISTS=false
        echo "Disaster Recovery partition does not exist" >&107
    fi
}

function pd_recreate_dr_efi_part {
    DR_SCRIPT="/opt/cisco/hostos/bin/dr_create_part_on_rp.sh"
    local device=/dev/mapper/${1}
    local mount_dir=${HOSTOS_DISK1_PATH}

    fsck -y ${device} >&107 2>&1
    /sbin/tune2fs -c 0 -i 0 ${device} >&107 2>&1
    mkdir -p ${mount_dir}
    rm -rf ${mount_dir}/*
    mkdir -p $HOSTOS_DISK1_PATH
    mount -o discard ${device} ${mount_dir} >&107 2>&1
    if [ -f ${mount_dir}/${REC_IMG_PATH}/${REC_IMG} ]; then
        if [ -f $DR_SCRIPT ]; then
            $DR_SCRIPT >&107
            echo "Check $DR_LOGFILE for DR partition creation logs" >&107
        fi
    else
        echo "Unable to locate Disaster Recovery image in ${mount_dir}/${REC_IMG_PATH}" >&107
    fi
    umount -f ${mount_dir} >&107 2>&1
}

function pd_check_if_obfl_on_disk {
    # Aug-4, we don't go for sda5 partition for OBFL...instead use obfl flash
    pd_is_augustus_arm_processor
    ret_val=$?
    if   pd_is_augustus_card && [  -b /dev/sda ] && [ "$ret_val" -eq 0 ]; then
            echo "yes" 
    else
        echo "no"
    fi                                                                          
}

function pd_is_disk_boot_arm_card
{
    if   pd_is_augustus_card ; then
        echo "TRUE"
    else
        echo "FALSE"
    fi
}

 
#augustus8 and augustus16 have disks. uboot doesn't understand gpt.
#so, we have to go legacy boot way
function pd_pxe_set_boottype {
    if  pd_is_augustus_card ; then
        #dmidecode -t processor | grep Manufacture | grep Intel > /dev/null 2>&1
        pd_is_augustus_arm_processor
        if [ "$?" -eq 0 ]; then
            BOOTTYPE="legacy"   #ARM Uboot
        else
            BOOTTYPE="efi"      #Intel EFI
        fi
    else 
        BOOTTYPE="efi"
    fi
}

function pd_is_augustus_arm_processor {
    dmidecode -t processor | grep Manufacture | grep Intel > /dev/null 2>&1
    if [ "$?" -ne 0 ]; then
        platform_log "Fabric card is ARM based..."
        return 0
    else
        platform_log "Fabric card is Intel based..."
        return 1
    fi
}

function update_uboot_host_boot_partition {
    new_boot_part=$1
    current_boot_part=$(findfs LABEL=HostOs)
    if [ "${new_boot_part}" != "${current_boot_part}" ]; then
        #TODO: add a retry mechanism
        tune2fs -L HostOs_Backup "${current_boot_part}"
    fi 
    tune2fs -L HostOs "${new_boot_part}"
}

function remove_uboot_backup_boot_partition {
    backup_part=$(findfs LABEL=HostOs_Backup)
    if [ -n "${backup_part}" ]; then 
        tune2fs -L "" ${backup_part}
    fi
}

function update_uboot_host_backup_boot_partition {
    new_backup_part=$1
    current_backup_part=$(findfs LABEL=HostOs_Backup)
    if [ "${new_backup_part}" != "${current_backup_part}" ]; then
        #add a retry mechanism
        tune2fs -L "" "${current_backup_part}"
    fi 
    tune2fs -L HostOs_Backup "${new_backup_part}"
}

function revert_uboot_host_boot_partition {
    current_boot_part=$(findfs LABEL=HostOs)
    current_backup_part=$(findfs LABEL=HostOs_Backup)
    if [ -n "${current_backup_part}" -a  -b "${current_backup_part}" ]; then
        #if we have a backup partition, mark it active
        tune2fs -L "" "${current_boot_part}"
        tune2fs -L HostOs "${current_backup_part}"
    fi
}

function pd_get_boot_dev {
    declare -F pd_pxe_set_boottype &>/dev/null && pd_pxe_set_boottype
    if [ "x${BOOTTYPE}" == "xlegacy" -a  -b /dev/sda1 ] ;then
        echo "/dev/sda1"
    else     
        echo "/dev/sda4"
    fi
}

# function to check augustus card indices
function pd_is_augustus_card {
    get_card_index                                                              
    [ "${CARDINDEX}" -eq 27037 ] || [ "${CARDINDEX}" -eq 27042 ] || [ "${CARDINDEX}" -eq 27043 ]
}

# function to check eyrie RP card indices
function pd_is_eyrie_rp_card {
    get_card_index                                                              
    [ "${CARDINDEX}" -eq 27065 ]
}
function get_boot_mode {
    boot_reg_val=$(iofpga_reg_read 0x40)
    case "${boot_reg_val}" in
    0x00000000)
        echo "Diskboot"
        ;;
    0x00000001)
        echo "Netboot"
        ;;
    *)
        echo "Netboot"
    esac    
}

# Usage : pd_get_ignore_file_list CARD_INSTANCE

function pd_get_ignore_file_list () {

    file_list=()
    file_list[0]="libesd_bcmsdk.so"

    IFS=',' read -r -a inst_array <<<${1}
    #Array for supported card instances corresponding to a file
    #Remove extension while creating *_card_instances variable for a file
    declare -a libesd_bcmsdk_card_instances=("FRETTA-LC" "FRETTA-HYBRID-LC" "FRETTA-SYNC-RP" "FRETTA-FC" "FRETTA-XC" "FRETTA-SYNC-RP2")

    ignore_list=""

    for file in "${file_list[@]}"
    do
        file_base_name="${file%%.*}"
        card_inst_arr=${file_base_name}_card_instances[@]
        bFound=false

        for instance in "${!card_inst_arr}"
        do
            for ci in "${inst_array[@]}"
            do
                if [[ ${ci} == ${instance} ]]
                then 
                    bFound=true
                    break
                fi
            done
            if [[ ${bFound} = true ]]
            then
                break
            fi
        done

        if [[ ${bFound} = false ]]
        then
            ignore_list="${ignore_list}${file} "
        fi
    done

    echo ${ignore_list}
}

#----------------------------------------------------------------------
# PI Diskmon monitor primary/secondry disk volume health and notify 
# failure to PD layer. Failure can be fatal or error event.
# Action :
#         1) Fatal event : Inform user and reload board.
#         2) Error event : Inform user and keep board in same state
#                          for debug.
#Failures are classified as follows:
#Primary volume:
#   - Missing: fatal event
#   - Health check failure: h/w error event
#   - ATA failed: fatal event
#Other volumes:
#   - All failures: h/w error event
#----------------------------------------------------------------------
function pd_notify_hw_fatal_event () {
    pd_send_post_code_to_sc $POST_CODE_HOST_DISK_FATAL_EVENT "Primary volume missing or health check failed"
}

function pd_notify_hw_error_event () {
# No secondary volume in ncs5500 so don't send any post code.
# Disable it for now.
    #pd_send_post_code_to_sc $POST_CODE_HOST_DISK_ERROR_EVENT "Secondary volume health check failed"
    platform_log "No Secondary volume in fretta. Ignore it."
}

function pd_is_install_tmpfs_supported () {
    if [ "$BOARDTYPE" == "RP" ] || [ "$BOARDTYPE" == "LC" ] || pd_is_augustus_card ; then
        return 1 
    fi
    # 1 for Supported : for RP LC and Augustus card
    # 0 for Not supported : for membooted card
    return 0
}

# Optimize card baking process, by remvoing unwanted iso 
# on the ramdisk.
# For Augustus-4, we dont want to copy xr.iso to install_repo 

function pd_remove_unwanted_iso () {

   if  pd_is_augustus_card ; then
        pd_is_augustus_arm_processor
        if [ "$?" -ne 0 ]; then
            platform_log_console "FC card is Intel based, delete xr.iso image from ramdisk..."

            platform_log_console "Removing iso image /iso/$SDR_ISO..."
            rm /iso/$SDR_ISO
        fi
    fi
}
