#!/bin/bash 
#
# calvados/zermatt_pkg/boot/scripts/spirit_pd.sh
# This script provides PD variables to other scripts.
#
# Copyright (c) 2014-2021 by Cisco Systems, Inc.
# All rights reserved.

# source PD-function library

if [ -f /etc/rc.d/init.d/pd-functions ]; then
    source /etc/rc.d/init.d/pd-functions
fi

if [ -f /etc/rc.d/init.d/mod_ins/module-load-functions ]; then
    source /etc/rc.d/init.d/mod_ins/module-load-functions
fi

if [ -f /etc/init.d/get_card_inst.sh ]; then
    source /etc/init.d/get_card_inst.sh
fi

if [ -f /opt/cisco/hostos/bin/part_script.sh ]; then
    source /opt/cisco/hostos/bin/part_script.sh
fi

#ARCH need to define here so that in kdump script COMPRESS flag set 
#and avoid INCOMPLETE.kdump issue 
ARCH=`arch`

##############################################################
# PCI INIT SECTION CODE BEGIN
# ############################################################
#
function init_iofpga_base_addr () {
    init_iofpga_base_addr_zermatt
}

# Init the values at start of script 
init_iofpga_base_addr

function iofpga_reg_write () {
    iofpga_reg_write_zermatt $1 $2 $3   
}

function iofpga_reg_read () {
    iofpga_reg_read_zermatt $1 $2    
}

function pd_is_install_tmpfs_supported () {
    return 1 
}

#
# Wrapper functions for accessing IOFPGA registers
# iofpga_reg_read/iofpga_reg_write take one more parameter on fixed platform
#
function iofpga_reg_write_common () {
    iofpga_reg_write $1 $2 $3
}

function iofpga_reg_read_common () {
    iofpga_reg_read $1 $2
}

##############################################################
# PCI INIT SECTION CODE ENDS 
# ############################################################

local_vmtype="$(cat /proc/cmdline|awk -F"vmtype=" '{print $2}'|cut -d' ' -f1)"
if [ ! "$local_vmtype" == "xr-vm" ]; then
    source /etc/init.d/load_kernel_modules.sh
fi

BOOTSTRAP_FILE="/etc/init.d/calvados_bootstrap.cfg"
source $BOOTSTRAP_FILE

function is_simulation_platform()
{
    return 0
}

function on_sim()
{
    return 0
}

#--- Set this PLATFORM_HOST_BRIDGE variable to setup bridge on host
#--- Reset PLATFORM_BRIDGE_EXT_CONNECTIVITY so that the bridge is not
#--- connected to an external interface. Both these variables are
#--- interpreted by nwbridge script.
if [[ $PLATFORM_HOST_BRIDGE = "" ]]; then
    readonly PLATFORM_HOST_BRIDGE=1
fi

if [[ $PLATFORM_BRIDGE_EXT_CONNECTIVITY = "" ]]; then
    readonly PLATFORM_BRIDGE_EXT_CONNECTIVITY=0
fi

function get_board_type {
    get_board_type_zermatt
}

# Nothing to override in Zermatt platform
function override_platform_type {
   echo $1
   return
}

#
# function to move ahci interrupt to last processor core
#
function move_irq_for_ahci() {
    NUM_PROC=$(grep -c ^processor /proc/cpuinfo)
    if [[ $NUM_PROC -le 4 ]]; then
        # Too few processors
        return
    fi

    if [[ $(ls -d /proc/irq/*/ahci 2>/dev/null | wc -l) -eq 0 ]]; then
        # No ahci devices
        return
    fi

    platform_log "======== begining move_irq_for_ahci for ${HOST_NAME}.. ========"

    # Set default affinity to skip last core
    # Assume less than 64 cores
    mask=$(printf %x $(((1 << ($NUM_PROC - 1)) - 1)))
    echo $mask > /proc/irq/default_smp_affinity

    platform_log "======== set mask to default_smp_affinity, $mask ========"

    # All ahci goes to last processor core
    for ahci in /proc/irq/*/ahci; do
        echo $(($NUM_PROC - 1)) >> $(dirname $ahci)/smp_affinity_list
    done

    for irq in /proc/irq/[0-9]*; do
        # On x86-based systems, first 16 irqs are typically handled by 
        # ioapic so there is no real need to move them
        # On current arm-based systems, it won't reach this point since
        # they have at most 4 cores 
        if [ $(basename $irq) -lt 16 ]; then continue; fi
        smp_mask=$((0x`cat $irq/smp_affinity` & 0x$mask))

    platform_log "======== in for loop, smp_mask = $smp_mask ========"

        if [ $smp_mask -ne 0 ]; then
            printf %x $smp_mask >  $irq/smp_affinity
        fi
    done

    platform_log "======== at the end of move_irq_for_ahci for ${HOST_NAME}.. ========"
}

function platform_specific_configuration() {
    local VM_TYPE=$1
    local VIRT_METHOD=$2
    local PLATFORM=$3
    get_board_type

    platform_log "platform specific configuration for $VM_TYPE"

    if [ "$VMTYPE" == "xr-vm" ]; then
        mount -o remount -o size=1M /dev
        ifconfig eth-mgmt up
    fi

    if [ "$VM_TYPE" == "hostos" ]; then
        set_proc_sys_net

        # to move ahci interrupt to last processor core
        move_irq_for_ahci
    fi

    if [ "$VM_TYPE" == "sysadmin-vm" ]; then
        # Punch watchdog for first time in sysadmin-vm, this will give enough
        # time for WDMON to take over on the handling of the IOFPGA watchdog
        declare -F pd_punch_watchdog && pd_punch_watchdog

        # Let Master RP know that SysAdmin VM has booted
        declare -F pd_notify_sysadmin_vm_booted && pd_notify_sysadmin_vm_booted

    fi
    # copy static host file to sysadmin of node
    cp /opt/cisco/hostos/etc/fretta_fixed_hosts /etc/hosts

    if [ "$VM_TYPE" != "hostos" ]; then
        return
    fi

    # Punch watchdog before loading kernal modules
    declare -F pd_punch_watchdog && pd_punch_watchdog

    load_all_kernel_module

    . /etc/init.d/nwbridge_pd.sh

    # ZERMATT_BRINGUP_HACK_luislu 
    # what is the hack: enable watchdog
    # why is the hack : CCTRL KLM disable watchdog after it is loaded.
    #                   This create a windows without watchdog until it is
    #                   re-enabled by WDMON from SysAdmin VM.
    # what is required to remove the hack : new CCTRL KLM without the
    #                   watchdog disabling behavior.
    local wdog_ctrl_reg=$(iofpga_reg_read 0 0x60)
    # if [ "$wdog_ctrl_reg" == "0x000000FF" ]; then
    #     echo "Re-enabling IOFPGA watchdog with timeout of 90 seconds"
    #     iofpga_reg_write 0 0x60 0x2D
    # fi

    # ZERMATT_BRINGUP_HACK_luislu 
    # what is the hack: Remove tcf-agent start and kill scripts
    # why is the hack : This software component is not used at all and it
    #                   cause more than 15 seconds of delay in graceful
    #                   shutdown case.
    # what is required to remove the hack : new OS SDK wil this removed.
    rm -f /etc/rc*.d/*tcf-agent

    HOST_NAME="host:0_RP0"
    hostname "$HOST_NAME"
   
    baseaddr=0x$(grep MMCONFIG /proc/iomem | awk '{print $1}' | sed -e "s;-.*;;") 
    stdbuf -oL /opt/cisco/hostos/bin/linkmonhost_ncs $baseaddr > /var/log/linkmonhost.log 2>&1 &
      
    echo -n "done platform_specific_configuration for ${HOST_NAME}.."
    return
}



#----Are we on simulation platform?----
if [ -z $SIMULATION ]; then
    readonly SIMULATION=0
fi

#------------------- Install patch files ---------#
if [ -f /etc/init.d/xrnginstall-zermatt-pd.sh ]; then
    source /etc/init.d/xrnginstall-zermatt-pd.sh
fi

# Zermatt platform specific hostOS patching
function  xrnginstall_platform_patch_hostos()
{
    local PARTMNT=$1
    local LVG_NAME=$2
    local HOST_LV_NAME=$3
    local PLATFORM=$4
    local SIM=$5
    local VIRT_METHOD=$6

    platform_log "$FUNCNAME"
    platform_log_console "ZERMATT: Patch host"
    xrnginstall_zermatt_patch_hostos $*
    platform_log_console "ZERMATT: Patch host done"
}

# Zermatt platform specific calvados patching
function xrnginstall_platform_patch_calvados()
{
    local PARTMNT=$1
    local PLATFORM=$2
    local SIM=$3
    local VIRT_METHOD=$4

    platform_log "$FUNCNAME"
    platform_log_console "ZERMATT: Patch calvados"
    xrnginstall_zermatt_patch_calvados $*
    platform_log_console "ZERMATT: Patch calvados done"
}

# Zermatt platform specific XR boot partition patching
function xrnginstall_platform_patch_xr()
{
    local XROOT=$1
    local PLATFORM=$2
    local SIM=$3
    local VIRT_METHOD=$4

    source /etc/rc.d/init.d/pd-functions

    platform_log "Patch XR for Zermatt"

    platform_log_exec cat /proc/cmdline

    xrnginstall_zermatt_patch_xr $XROOT $PLATFORM

    platform_log "Patch XR for Zermatt done"
}

# Only cards running x86 CPU support this, so this is applicable only for
# RP's
# RP BIOS_BOOT_MODE register is at address 0x50
RP_IOFPGA_BIOS_BOOT_MODE_REG_ADDR=0x50
function pd_reset_bios_boot_mode () {
     echo "Resetting RP BIOS_BOOT_MODE register"
     iofpga_reg_write 0 $RP_IOFPGA_BIOS_BOOT_MODE_REG_ADDR 0
}

#
# Punch the IOFPGA watchdog
IOFPGA_SYS_WDOG_RESET_REG_ADDR=0x64
IOFPGA_SYS_WDOG_RESET_KICK_WDOG_BIT=0x80000000
function pd_punch_watchdog () {
   echo "Punching IOFPGA watchdog"
   iofpga_reg_write 0 $IOFPGA_SYS_WDOG_RESET_REG_ADDR $IOFPGA_SYS_WDOG_RESET_KICK_WDOG_BIT
}
 
#
# Configure the IOFPGA watchdog with a specific stage1 timeout value.
# With Fretta there is only one timeout value that apply to both stage1 and
# stage2 watchdog timeout.
# Input value is in seconds, but when we write into the watchdog control
# register we have to divice it by 2 as each unit represent 2 seconds.
IOFPGA_SYS_WDOG_CTRL_REG_ADDR=0x60
function pd_cfg_watchdog_timeout () {
    local wdog_timeout=$(printf 0x%08X $(($1 / 2)))

    iofpga_reg_write 0 $IOFPGA_SYS_WDOG_CTRL_REG_ADDR $wdog_timeout
}

function pd_get_active_rp_slot () {
    echo "0"
}

function check_uart_skew () {

# FAB Revision Register value
    fab_register=0x8000054

# Get Address of CPU IOFPGA Device
    all_base_addresses=`lspci -vvd:0100 | grep Memory | cut -d " " -f5`
    
    # There are boards where more than one BAR address exists, 
    # Like TURIN-CR-MACSEC where 3 BARs are there, get  the first one
    base_address=`echo $all_base_addresses | cut  -b 1-8`

# Read FAB Revision Register
    fab_offset=`printf "%x" $((0x$base_address+$fab_register))`
    fab_value=`pcimemread $fab_offset 4 | cut -d " " -f11`
# Return 0 (not skewed) if revision greater than 2  
# Return 1 ( skewed) if revision is P0/P1/P2
# Tortin/Peyto only has 115200 baud-rate by default
    platf_type=$(iofpga_reg_read 0 4)  
    local num=$( printf "0x%x" $platf_type )
    local result=$(($(( num >> 23 & 0x1 )) << 4 | $(( num >> 28 & 0xF ))))
    case ${result} in
        7)  echo "NCS5501-HD: Baud-rate set 115200" 
	    return 0 
	    ;;  # Tortin
        23)  echo "NCS5501-HD: Baud-rate set 115200" 
	    return 0 
	    ;;  # Tortin 16G
        10)  echo "NCS-55A2-MOD-SE-S: Baud-rate set 115200" 
	    return 0 
	    ;;  # Peyto TCAM
        11)  echo "NCS-55A2-MOD-S: Baud-rate set 115200" 
	    return 0 
	    ;;  # Peyto Non TCAM
        13)  echo "NCS-55A2-MOD-HD-S: Baud-rate set 115200" 
	    return 0 
	    ;;  # Peyto Non TCAM HD
        15)  echo "NC55A2-MOD-SE-H-S: Baud-rate set 115200" 
	    return 0 
	    ;;  # Peyto TCAM with HD CC
        *)  ;; #for rest use h/w revision to decide
    esac

    echo "Zermatt/Turin: Hardware Revision: ${fab_value}"
    if [ $fab_value -gt 2 ] 
    then
        return 0 
    else
        return 1
    fi 
}

function get_num_npus () 
{
    local plat_type=$(iofpga_reg_read 0 4)
    local num=$( printf "0x%x" $plat_type )
    local result=$(($(( num >> 23 & 0x1 )) << 4 | $(( num >> 28 & 0xF ))))
    case ${result} in
        1) return 8 ;;  # Zermatt
        2) return 1 ;;  # TurinMx
        3) return 1 ;;  # Taihu
        4) return 8 ;;  # ZermattCR
        5) return 2 ;;  # Winterfell
        6) return 4 ;;  # OldCastle
        7) return 1 ;;  # Tortin
        8) return 2 ;;  # Pyke
        9) return 4 ;;  # OldCastleCR
        10) return 1 ;;  # Peyto 
        11) return 1 ;;  # PeytoCR
        12) return 2 ;;  # Bifrost-T
        13) return 1 ;;  # PeytoCRHD
        14) return 1 ;;  # TurinCR
        15) return 1 ;;  # PeytoCCHd
        23) return 1 ;;  # Tortin 16G
        *)
           return 0 ;;
    esac
}

#NJ_TBD : Add for TURIN_CR once FPGA_ID is available
function num_psinb_on_bcm_knet () 
{
    local plat_type=$(iofpga_reg_read 0 4)
    local num=$( printf "0x%x" $plat_type )
    local result=$(($(( num >> 23 & 0x1 )) << 4 | $(( num >> 28 & 0xF ))))
    case ${result} in
        1) return 0 ;;  # Zermatt
        2) return 0 ;;  # TurinMx
        3) return 1 ;;  # Taihu
        4) return 0 ;;  # ZermattCR
        5) return 2 ;;  # Winterfell
        6) return 4 ;;  # OldCastle
        7) return 0 ;;  # Tortin
        8) return 2 ;;  # Pyke
        9) return 4 ;;  # OldCastleCR
        10) return 1 ;;  # Peyto 
        11) return 1 ;;  # PeytoCR
        12) return 2 ;;  # Bifrost-T
        13) return 1 ;;  # PeytoCRHD
        14) return 1 ;;  # TurinCR
        15) return 1 ;;  # PeytoCCHd
        23) return 0 ;;  # Tortin 16G
        *)

           return 0 ;;
    esac
}

function get_card_index_from_fpga () {
    local platform_type=$(iofpga_reg_read 0 4)  
    local num=$( printf "0x%x" $platform_type )
    local result=$(($(( num >> 23 & 0x1 )) << 4 | $(( num >> 28 & 0xF ))))
    local card_index=0
    case ${result} in
        1) card_index=27014     # Zermatt
           ;;
        2) card_index=27016     # Turin
           ;;
        3) card_index=27020     # Taihu
           ;;
        4) card_index=27019     # Zermatt-CR
           ;;
        5) card_index=27021     # Winterfell
           ;;
        6) card_index=27025     # Oldcastle
           ;;
        7) card_index=27027     # Tortin no TCAM, without MACSEC
           ;;
        8) card_index=27029     # Pyke No TCAM
           ;;
        9) card_index=27026     # OldcastleCR No TCAM
           ;;
        10) card_index=27030     # Peyto- with TCAM
           ;;
        11) card_index=27031     # PeytoCR - no TCAM
           ;;
        12) card_index=27047     # Bifrost-T - no TCAM 
           ;;
        13) card_index=27045     # PeytoCR - no TCAM HD
           ;;
        14) card_index=60004     # TurinCR - no TCAM 
           ;;
        15) card_index=60001     # Peyto- with TCAM HD CC  
           ;;
        23) card_index=27027     # Tortin16G no TCAM, without MACSEC
           ;;  
        *) platform_log_console "platform unknown"
           card_index=0
           ;;
    esac
    echo "$card_index"
}


function pd_pxe_update_efi_grub () {
    local input_dir=$1
    local grubcfg=${input_dir}/grub.cfg

    local card_index="$(get_card_index_from_fpga)"

    
    # Find out if UART is skewed or not
    check_uart_skew
    SKEW=$?
    if [ $SKEW -eq 1 ]
    then
         sed -i -e "s;console=ttyS1,115200;console=ttyS0,38400 8250.uartclk=2457600;" \
         -e "s;--speed=115200;--speed=38400;" ${grubcfg}
    else
         sed -i -e "s;console=ttyS1,115200;console=ttyS0,115200;" ${grubcfg}      
    fi

    sed -i -e "s;cgroup_disable=memory;card_index=${card_index} mtdparts=physmap-flash.0:256k(SM_LOG),256k(RR_LOG),256k(KLOG),18m(PLOG),512k(mtdoops),4m(shmoo) printk.always_kmsg_dump=1 printk.kmsg_dump_only_panic=0 intel_idle.max_cstate=0;" \
    -e "s;crashkernel=256M@0;crashkernel=400M@0 slub_debug=-;" \
    -e "s;bigphysarea=10M;bigphysarea=256M;" ${grubcfg}

    #Set the swiotlb size to 128MB (slab size is 2k => 64k slabs)
    #if the number of ps inb interfaces are greater than 2
    local num_knet_psinb
    num_psinb_on_bcm_knet 
    num_knet_psinb=$?
    if [ $num_knet_psinb -gt 2 ]; then
        sed -i -e '/linux / s/$/ swiotlb=0x10000/' ${grubcfg}
    fi
    platform_log_console "ZERMATT: Patch grub. card_index=${card_index}"
    mv ${input_dir} ${input_dir}/../boot/
}

function pd_pxe_update_pxeboot_iso {
    # Nothing is needed to be done here as we do not have other cards
    return
}


# populates the variable DEVVDB1 & VDB1P1
# $1 should be DEVVDB1
# $2 should be VDB1P1
function xrnginstall_platform_set_vdb
{
    if [[ $# -ne 2 ]]; then
        echo "$FUNCNAME: Wrong argument list"
        return 1
    fi
 
    eval $1="/dev/vdb1"
    eval $2="vdb1p1"
}

# Load the platform specific kernel modules here
function platform_load_kernel_module () {
    return
}

function platform_copy_required_kernel_crash_modules() {
    local dest=$1

    # Copying all required modules
    cp /lib/modules/3*/kernel/drivers/md/dm-mod.ko $dest

    return
}

function platform_load_required_kernel_crash_modules() {

  insmod /lib*/modules/dm-mod.ko
}

function xrnginstall_platform_modprobe_uio_dma_proxy
{
    local BOARDTYPE=`cat /proc/cmdline | grep boardtype=`
    if [ -n "$BOARDTYPE" ]; then
        BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    fi
 
    if [ "$BOARDTYPE" = "LC" ]; then
        modprobe uio_dma_proxy dmainfo="mvdma0=1M,0 mvdma1=1M,1M \
        mvdma2=1M,2M FD_LC=6M,3M"
    else
        modprobe uio_dma_proxy dmainfo="mvdma0=1M,0 mvdma1=1M,1M \
        mvdma2=1M,2M Fabric_SDK=10M,3M Fabric_FGID=40M,13M FD_RP_FC=45M,53M"
    fi
}

#
# IOFPGA bootloader control status register
# When bit 12 is set it indicates that the backup image is booted
IOFPGA_BOOTLDR_CNTRL_STATUS_REG_ADDR=0x68
IOFPGA_BOOTLDR_BACKAUP_ACTIVE_BIT=0x1000

#
# IOFPGA bootloader running version register
IOFPGA_BOOTLDR_RUN_VER_REG_ADDR=0x10


# Send HOST_STARTED event  : 
# NOTE: We are not notifying any entity here 
IOFPGA_REG_RESET_REASON=0x88
function pd_notify_host_started () {
   # Show last reset reason register value
   reset_reason=$(iofpga_reg_read 0 $IOFPGA_REG_RESET_REASON)
   echo "Last Reset Reason =  $(printf 0x%08X $reset_reason)"

   # Show watchdog register value
   wdog_reg=$(iofpga_reg_read 0 0x60)
   echo "Watchdog Register =  $(printf 0x%08X $wdog_reg)"
}

#
# Configure the IOFPGA status LED to turn solid amber color. This is achieved
# by turning on both red and green color at the same time.
function pd_set_status_led_to_solid_amber() {
    local reg_addr=0x80
    local iofpga_instance
    local card_index
    local plat_type=$(iofpga_reg_read 0 4)
    local num=$( printf "0x%x" $plat_type )
    local result=$(($(( num >> 23 & 0x1 )) << 4 | $(( num >> 28 & 0xF ))))
    case ${result} in
        1) ;&                    # Zermatt
        4) iofpga_instance=2 ;;  # Zermatt-CR
        2) ;&                    # Taihu
        5) ;&                    # Winterfell
        6) ;&                    # Oldcastle
        7) ;&                    # OldcastleCR No TCAM
       12) ;&                    # Bifrost-T ; 
       14) ;&                    # TurinCR ; 
       23) ;&                    # Tortin 16G 
        3) iofpga_instance=1 ;;  # Turin
        *) platform_log_console "platform unknown"
           iofpga_instance=1 ;;
    esac

    local old_val=$(iofpga_reg_read $iofpga_instance $reg_addr)

    # set STS LED bits to solid amber
    local new_val=$(($old_val & 0xc3ffffff))
    new_val=$(($new_val | 0x14000000))

    new_val=$(printf 0x%08X $new_val)

    iofpga_reg_write $iofpga_instance $reg_addr $new_val
}

#
# Send SYSADMIN_VM_STARTED : 
# NOTE: We are not notifying any entity here 
function pd_notify_sysadmin_vm_started () {
    # Show watchdog register value
    wdog_reg=$(iofpga_reg_read 0 0x60)
    echo "Watchdog Register =  $(printf 0x%08X $wdog_reg)"

    # Turn the status LED to amber to indicate that host is ready
    pd_set_status_led_to_solid_amber
}
RP_IOFPGA_PWR_CYCLE_CTRL_REG_ADDR=0x74
IOFPGA_BOARD_CONTROL_REG_ADDR=0x300
#
# Called to perform board reset operation
function pd_board_reset () {
   local reg_addr=0x0
   local reg_val=0x0

       reg_addr=$RP_IOFPGA_PWR_CYCLE_CTRL_REG_ADDR
       reg_val=0x1

   if [ "$reg_val" != "0" ]; then
       echo "Resetting board via IOFPGA reg:$reg_addr val:$reg_val..."
       iofpga_reg_write 0 $reg_addr $reg_val
   fi
}

#
# Calculate the offset for the partition 
function get_fs_offset {
    local start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $2}')
    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')
    fi
    local offset=$(expr ${start} \* 512)
    echo ${offset}
}

#
# Calculate the sizelimit for the partition 
function get_fs_sizelimit {
    local result
    local sizelimit
    local start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $2}')
    local endlimit=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')

    if [ "${start}" == "*" ]; then
        start=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $3}')
        endlimit=$(fdisk -u -l ${1} | grep "${1}${2}" | awk '{print $4}')
    fi
    result=$((${endlimit} - ${start}))
    sizelimit=$(expr ${result} \* 512)
    echo ${sizelimit}
}

#
# Get my nodeid string in internal format (e.g. 0_RP0)
function pd_get_my_int_nodeid_str () {
    echo "0_RP0"
}

#
# Get my nodeid string in external format (e.g. 0/RP0)
function pd_get_my_ext_nodeid_str () {
    echo "0/RP0"
}

#
# Usage : pd_get_card_inst card_inst
function pd_get_card_inst () {
    local lcard_inst
    get_board_type

    pd_get_card_inst_from_card_type_zermatt $BOARDTYPE lcard_inst

    eval "$1=$lcard_inst"
}

function check_for_ssd_vol_resize
{
    lv_name=$1
    new_lv_size=$2
    local total_disk_size=$3
    
    gib_in_mb="1073.74"

    cur_lv_size=`lvdisplay $lv_name | grep "LV Size" | awk '{print $3}'`
    perc=`echo "$cur_lv_size $gib_in_mb $total_disk_size" | awk '{print int($1 * $2 * 100 / $3)}'`

    if [ $perc -eq $((new_lv_size-1)) ] ||
       [ $perc -eq $new_lv_size ] ||
       [ $perc -eq $((new_lv_size+1)) ]; then
        return 0 
    else
        return 1
    fi
}

function pd_check_for_secondary_vols_resize
{
    dev=$1
    
    host_ssd_lv="/dev/${DISK1_PV}/${DISK1_LV}${D1_HOST_LV}"
    cal_ssd_lv="/dev/${DISK1_PV}/${DISK1_LV}${D1_CVDS_LV1}"
    xr_ssd_lv="/dev/${DISK1_PV}/${DISK1_LV}${D1_XR_LV1}"

    get_disk_size ${dev}

    check_for_ssd_vol_resize $host_ssd_lv $DISK_SSD_HOST_SIZE $DISK_SIZE
    if [ $? -ne 0 ]; then
        return 1
    fi
    check_for_ssd_vol_resize $cal_ssd_lv $DISK_SSD_CALVADOS_SIZE $DISK_SIZE
    if [ $? -ne 0 ]; then
        return 1
    fi
    check_for_ssd_vol_resize $xr_ssd_lv $DISK_SSD_XR_SIZE $DISK_SIZE
    if [ $? -ne 0 ]; then
        return 1
    fi

    return 0
}

# Enable panic on Oops and panic on softlockup.
# panic on hardlockup is not configurable.
function pd_set_kernel_panic_triggers
{
    echo 1 > /proc/sys/kernel/panic_on_oops
    echo 1 > /proc/sys/kernel/softlockup_panic
}

# Check if Shmoo Partition is needed
function pd_is_shmoo_partition_needed
{
    if [ "$BOARDTYPE" == "RP" ];then
        return 1
    else
        return 0
    fi
}

#
# RP BIOS_BOOT_MODE register is at address 0x50.
# Find boot mode(ipxe/usb)
# ipxe == 2 and usb == 4
RP_DISK_BOOT_VAR=0x00000000
RP_IPXE_BOOT_VAR=0x00000001
RP_EPXE_BOOT_VAR=0x00000002
RP_USB_BOOT_VAR=0x00000004
function pd_get_boot_mode () {
    if [ "${BOARDTYPE}" == "RP" ]; then
        local boot_dev=""
        local platform=$(cat /proc/cmdline | awk -F "platform=" ' { print $2 }' | cut -d " "  -f1)
        local boot_mode_cmdline="$(cat /proc/cmdline|awk -F"boot_mode=" '{print $2}'|cut -d' ' -f1)"
        local boot_mode=""
        if [ ! -z "$boot_mode_cmdline" ]; then
            boot_mode=$(printf 0x%08X $boot_mode_cmdline)
        else
            init_iofpga_base_addr
            boot_mode=$(iofpga_reg_read 0 $RP_IOFPGA_BIOS_BOOT_MODE_REG_ADDR)

            # We are in the pxe_install.sh context. if user triggered boot from 
            # BIOS prompt then boot_mode will have value as 0 (default initialised)
            # so 0 is treated as DISK_BOOT which is not the case here. 
            # Here boot_mode is overwritten to RP_EPXE_BOOT_VAR/RP_IPXE_BOOT_VAR/  
            # RP_USB_BOOT_VAR based on platform/active or standby node/boot_mode 
            # in cmdline for GISO USB boot
            
            if [ $boot_mode == $RP_DISK_BOOT_VAR ]; then
                if [ "${platform}" == "zermatt" ]; then
                    boot_mode=$RP_EPXE_BOOT_VAR
                else
                    local im_active=$(pd_check_rp_im_active)
                    if [ "$im_active" == "1" ]; then
                        boot_mode=$RP_EPXE_BOOT_VAR
                    else
                        boot_mode=$RP_IPXE_BOOT_VAR
                    fi
                fi
            fi
        fi

        if [ $boot_mode == $RP_EPXE_BOOT_VAR ]; then
            boot_dev="EPXE"
        elif [ $boot_mode == $RP_USB_BOOT_VAR ]; then
            boot_dev="USB"
        else
            boot_dev="IPXE"
        fi
        echo "${boot_dev}"
    fi
}

function pd_handle_swtam {
    mkdir -p /misc/config/swtam
    ln -sf /misc/config/swtam /misc/swtam
    platform_log_console "Softlink created /misc/swtam -> /misc/config/swtam"
}


function pd_dhclient_setup {
if [ -f /etc/dhclient-exit-hooks ]; then
    rm /etc/dhclient-exit-hooks
fi
cat <<EOF > /etc/dhclient-exit-hooks
#!/bin/bash

if [[ \${new_bootfile_name} = *"http"* ]]; then
  url=\${new_bootfile_name}
  echo Downloading file using bootfile-name: \${url}
  /usr/bin/wget -q \${url} -O $1
elif [ -n "\${new_dhcp_server_identifier}" -a -n "\${new_bootfile_name}" ]; then
  echo Downloading file using bootfile-name: \${new_bootfile_name}
  tftp_path=\`echo \${new_bootfile_name} | sed "s/tftp:\/\///g" | sed "s/^[^\/]*\///g"\`
  echo tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path}
  /usr/bin/tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path} $1
elif [[ \${new_filename} = *"http"* ]]; then
  url=\${new_filename}
  echo Downloading file using filename: \${url}
  /usr/bin/wget -q \${url} -O $1
elif [ -n "\${new_dhcp_server_identifier}" -a -n "\${new_filename}" ]; then
  echo Downloading file filename: \${new_filename}
  tftp_path=\`echo \${new_filename} | sed "s/tftp:\/\///g" | sed "s/^[^\/]*\///g"\`
  echo tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path}
  /usr/bin/tftp -v -m binary \${new_dhcp_server_identifier} -c get \${tftp_path} $1
fi
EOF
echo "timeout 10;" >> /etc/dhcp/dhclient.conf
}

function pd_get_serial_number {
    local serial_number=""
    serial_number=`dmidecode -s system-serial-number | grep -v "#"`
    echo $serial_number
}

function pd_get_product_name {
    local product_name=""
    product_name=`dmidecode -s system-product-name | grep -v "#"`
    echo $product_name
}

function pd_dhclient_conf {
  local product_name=$(pd_get_product_name)
  local serial_number=$(pd_get_serial_number)
  platform_log_console "Product name: $product_name"
  platform_log_console "Serial number: $serial_number"
  echo "send vendor-class-identifier \"PXEClient:Arch:00009:UNDI:003010:PID:$product_name\";" >> /etc/dhcp/dhclient.conf;
  echo "send user-class \"iPXE-GISO\";" >> /etc/dhcp/dhclient.conf;
  echo "send dhcp-client-identifier \"$serial_number\";" >> /etc/dhcp/dhclient.conf;
}

function pd_dhclient_conf_append_option () {
  local option=$1
  platform_log "Appending Option = $option"
  # In /etc/dhcp/dhclient.conf there are "request" present for other things like
  # subnet-mask, broadcast-address, etc. To add any additional option we need
  # to prepend with "also" keyword to "request" to make it additonal request
  # otherwise only "request" will override older request parameter 
  echo "also request $option ;" >> /etc/dhcp/dhclient.conf;
}

function pd_get_linkstatus() {
    local intf_name=$1
    local link_status=""

    link_status=`ethtool $intf_name | grep "Link detected" | cut -d ":" -f2`
    echo $link_status
}

function pd_copy_from_usb {
    local usbpath=""
    local usbmnt=""
    local removable_drives=""
    local usbgrub=""
    local usbiso=""
    local isofile=$1

    for _device in /sys/block/*/device; do
        if echo $(readlink -f "$_device")|egrep -q "usb"; then
            _disk=$(echo "$_device" | cut -f4 -d/)
            removable_drives="$_disk"
        fi
    done

    usbpath=`blkid -c /dev/null | grep $removable_drives | cut -d ":" -f1`
    platform_log_console "USB path: $usbpath"
    usbmnt=$(mktemp -d /tmp/usbdev.XXXXXX)
    mount $usbpath $usbmnt >&107 2>&1

    if [ -f "$usbmnt/EFI/boot/grub.cfg" ]; then
        usbgrub=$usbmnt/EFI/boot/grub.cfg
    elif [ -f "$usbmnt/boot/grub.cfg" ]; then
        usbgrub=$usbmnt/boot/grub.cfg
    fi

    if [ -n "$usbgrub" ]; then
        echo "Found grub file $usbgrub" >&107
        usbiso=$(grep loopback $usbgrub | head -1 | awk '{print $3}' | sed 's/\r$//')
        echo "Looking for $usbiso on $usbmnt" >&107
        if [ -f "$usbmnt$usbiso" ]; then
            usbiso=$usbmnt$usbiso
            echo "Found iso image at $usbiso" >&107
        else
           usbiso=
           echo "Cannot find iso image from grub.cfg" >&107
        fi
    fi

    if [ -z "$usbiso" ]; then
        if [ -f "$(echo $usbmnt/boot/*.iso*)" ]; then
          usbiso=$(echo $usbmnt/boot/*.iso*)
          echo "Found iso image at $usbiso" >&107
        elif [ -f "$(echo $usbmnt/*.iso*)" ]; then
          usbiso=$(echo $usbmnt/*.iso*)
          echo "Found iso image at $usbiso" >&107
        fi
    fi

    if [ -n "$usbiso" -a -f "$usbiso" ]; then
        cp $usbiso $isofile
    fi

    umount $usbmnt
    rmdir $usbmnt
}


function pd_download_giso {
    local boot_dev=""
    local system_image="system_image.iso"
	local max_dhclient_try=2
    local link_status=""

    mount -r -o loop /iso/$HOST_ISO ${ISOMNT} || exitclean
    local GIsoType=$(check_iso_type ${ISOMNT}) 
    umount ${ISOMNT} || exitclean
    platform_log_console "Golden ISO type = $GIsoType and PKG_VER = $PKG_VER"

    if [[ ${GIsoType} < ${PKG_VER} ]]; then
        platform_log_console "Detected non eXR platform. Exiting pd_download_giso"
        return
    fi

    if [ -f /iso/$system_image ]; then
        mount -r -o loop /iso/$system_image ${ISOMNT} || exitclean
        if [ -f ${ISOMNT}/giso_info.txt ]; then
            platform_log_console "Additional RPMS embedded in $system_image"
            cp /iso/${system_image} /${system_image}
            if (( $? != 0 )); then
                platform_log_console "Failed to copy $system_image"
            fi
            return
        fi   
    fi

    # fretta pd function
    declare -f pd_get_boot_mode >&107 2>&1                                       
    if [ "$?" -eq 0 ]; then                                                      
        boot_dev=$(pd_get_boot_mode)                                             
        if [ -z "${boot_dev}" ]; then                                            
            platform_log_console "Couldn't find boot id(iPXE/USB). Returning from pd_download_giso"
            return                                                               
        fi                                                                       
    fi 

    if [ ! -f /${system_image} ]; then
        # fretta standby rp 
        if [ "${boot_dev}" == "IPXE" ]; then                                      
            platform_log "Current Boot: Internal-iPXE"
            declare -F pd_int_pxe_download_iso >&107 2>&1 && \                
                pd_int_pxe_download_iso ./                                    
        elif [ "${boot_dev}" == "EPXE" ]; then                     
            platform_log "Current Boot: External-iPXE"
            pd_dhclient_setup /${system_image}
            pd_dhclient_conf
            pd_dhclient_conf_append_option "bootfile-name"

            local eths=( $(ifconfig -a | grep "Link\ " | cut -d' ' -f1 ) )
            for eth in "${eths[@]}";
            do
                ifconfig ${eth} up
            done
            sleep 5
            # Fetch ISO from PXE server.
            # Try on each interface.
            local dhc_try=0
            while [ ! \( -f /${system_image} -a `cat /${system_image} | wc -c` -gt 0 \) ]; do
                for eth in "${eths[@]}"
                do
                    link_status=$(pd_get_linkstatus ${eth})
                    if [ "$link_status" == "yes" ]; then
                        platform_log_console "Running dhclient over ${eth}"      
                        dhclient -1 ${eth}
                        if [ -f /${system_image} -a `cat /${system_image} | wc -c` -gt 0 ]; then
                            break 
                        fi
                    fi
                done

                dhc_try=$((dhc_try+1))
                if [ ${dhc_try} -gt ${max_dhclient_try} ]; then
                    platform_log_console "Failed to fetch ISO from DHCP server" 
                    break
                fi
            done
            # tftp download takes huge time for downlaoding the image
            # So to avoid timeout before completion of baking, 
            # watchdog time is increased by punching watchdog
            platform_log_console "Punching watchdog, downloading of whole ISO might time out" 
            declare -F pd_punch_watchdog && pd_punch_watchdog
        elif [ "${boot_dev}" == "USB" ]; then                     
            platform_log_console "Current Boot: USB"
            pd_copy_from_usb /${system_image}
        fi
    fi
}

#
# debug partition is needed
#
function pd_check_if_debug_part_needed {
    return 0
}

# Copy Recovery image from Host /misc/disk1
function pd_get_recovery_image {
    local boardtype="${1}"
    local isofile=system_image.iso
    local host_ssd_label=${DISK1_PV}-${DISK1_LV}${D1_HOST_LV}
    local host_ssd_dev=/dev/mapper/${host_ssd_label}

    rec_mnt_dir=$(mktemp -d /tmp/RECMNT.XXXXXX)
    mount -o discard ${host_ssd_dev} ${rec_mnt_dir} >&107 2>&1

    rec_img_path="$rec_mnt_dir/recovery"
    if [ ! -f $rec_img_path/$isofile ]; then
       platform_log_console "Unable to locate recovery image at $rec_img_path"
       ( cd $rec_img_path && ls -laR . ) >&107
       return 1
    fi

    platform_log_console "Located recovery image at $rec_img_path/$isofile"
    cp $rec_img_path/$isofile $isofile
    chmod 544 $isofile

    umount -f $rec_mnt_dir
    rm -rf $rec_mnt_dir
}

# Backup Disaster Recovery partition during bake phase if it exists
function pd_check_for_dr_efi_part {
    check_if_partitioned >&107 2>&1
    if [ $? == 1 ]; then 
        DR_EFI_EXISTS=true
        echo "Disaster Recovery partition exists already" >&107
    else
        DR_EFI_EXISTS=false
        echo "Disaster Recovery partition does not exist" >&107
    fi
}

function pd_recreate_dr_efi_part {
    DR_SCRIPT="/opt/cisco/hostos/bin/dr_create_part_on_rp.sh"
    local device=/dev/mapper/${1}
    local mount_dir=${HOSTOS_DISK1_PATH}

    fsck -y ${device} >&107 2>&1
    /sbin/tune2fs -c 0 -i 0 ${device} >&107 2>&1
    mkdir -p ${mount_dir}
    rm -rf ${mount_dir}/*
    mkdir -p $HOSTOS_DISK1_PATH
    mount -o discard ${device} ${mount_dir} >&107 2>&1
    if [ -f ${mount_dir}/${REC_IMG_PATH}/${REC_IMG} ]; then
        if [ -f $DR_SCRIPT ]; then
            $DR_SCRIPT >&107
            echo "Check $DR_LOGFILE for DR partition creation logs" >&107
        fi
    else
        echo "Unable to locate Disaster Recovery image in ${mount_dir}/${REC_IMG_PATH}" >&107
    fi
    umount -f ${mount_dir} >&107 2>&1
}


# Usage : pd_get_ignore_file_list CARD_INSTANCE

function pd_get_ignore_file_list () {

    file_list=()
    file_list[0]="libesd_bcmsdk.so"

    #Array for supported card instances corresponding to a file
    #Remove extension while creating *_card_instances variable for a file
    declare -a libesd_bcmsdk_card_instances=("ZERMATT-RP")

    ignore_list=""

    for file in "${file_list[@]}"
    do
        file_base_name="${file%%.*}"
        card_inst_arr=${file_base_name}_card_instances[@]
        bFound=false

        for instance in "${!card_inst_arr}"
        do
            if [[ ${1} == ${instance} ]]
            then
                bFound=true
            fi
        done

        if [[ ${bFound} = false ]]
        then
            ignore_list="${ignore_list}${file} "
        fi
    done

    echo ${ignore_list}
}

function pd_is_install_tmpfs_supported () {
    # 1 for Supported : for Zermatt
    return 1
}

function pd_clean_incomplete_kdumps () {
    #called by kdump script
    local calvados_ssd="/dev/mapper/pci_disk1-ssd_disk1_calvados_1"
    local calvados_ssd_mount_dir="/mnt/"

    # create the mount point
    mkdir -p ${calvados_ssd_mount_dir}

    # Initially we want to be as gentle as possible and hence we attempt to save
    # the incomplete kdumps to calvados in a hope that they are still useful. Moving
    # them makes sense only in case of fixed boxes and not in modular. In case of
    # modular, multiple SC can be creating dumps simulatneously on shared NFS and hence
    # moving them will break everything, so move only in case of fixed boxes.
    mount -o discard ${calvados_ssd} ${calvados_ssd_mount_dir} > /dev/null
    if [ $? -eq 0 ]; then
        # move *.kdump, *.INCOMPLETE.kdump, *.incomplete files to calvados
        echo "moving leftover kdumps (including incomplete kdumps) to calvados"
        echo "(Note: In case the move is unsuccessful, incomplete kdumps would be deleted)"

        # ${COREDIR} is defined by caller i.e. kdump script
        mv -f ${COREDIR}/*.kdump /mnt/ 2> /dev/null
        mv -f ${COREDIR}/*.incomplete /mnt/ 2> /dev/null
        umount ${calvados_ssd_mount_dir}
    else
        echo "mounting calvados disk partition failed, will delete all incomplete kdumps and proceed"
    fi
    # do not delete successful kdumps
    rm -f ${COREDIR}/*.INCOMPLETE.kdump
    rm -f ${COREDIR}/*.incomplete
}
