#!/bin/sh
# * Copyright (c) 2017-2021 by cisco Systems, Inc.
# * All rights reserved.
#
# Function for finding IOFPGA base address with lspci tool and set the
# 
# Instance 0 = CPU IOFPGA  (Applies to Zermatt/Turin IOFPGA system Only) 
# Instance 1 = BASE IOFPGA (Applies to Multi IOFPGA system like Zermatt/Turin) 
# Instance 2 = Mezz IOFPGA (Applies to Multi IOFPGA system like Zermatt/Turin)
#


if [[ "$GET_CARD_INST" == true ]]; then
    return
fi
GET_CARD_INST=true       

IOFPGA_VENDOR=1137
MIFPGA_VENDOR=1137
CPU_BOARD_IOFPGA_ID=0100
BASE_BOARD_IOFPGA_ID=0101
MEZZ_BOARD_IOFPGA_ID=0102
IOFPGA_OFFSET=0x08000000
MIFPGA_ID=0103

FRETTA_RP_SUPA_IOFPGA_VAL=0x01168153
FRETTA_RP_SUPB_IOFPGA_VAL=0x01423741
ZERMATT_RP_IOFPGA_VAL=0x10171FD3
ZERMATT_CR_IOFPGA_VAL=0x40171FD3
TURIN_RP_IOFPGA_VAL=0x20171FD3
TAIHU_RP_IOFPGA_VAL=0x30171FD3
WINTERFELL_RP_IOFPGA_VAL=0x50171FD3
OLDCASTLE_RP_IOFPGA_VAL=0x60171FD3
OLDCASTLE_CR_IOFPGA_VAL=0x90171FD3
TORTIN_RP_IOFPGA_VAL=0x70171FD3
TORTIN_RP_16G_IOFPGA_VAL=0x70971FD3
PEYTO_RP_IOFPGA_VAL=0xA0171FD3
PEYTO_CC_RP_IOFPGA_VAL=0xF0171FD3
PEYTO_CR_IOFPGA_VAL=0xB0171FD3
PEYTO_CR_HD_IOFPGA_VAL=0xD0171FD3
PYKE_RP_IOFPGA_VAL=0x80171FD3
MACSEC_18P_LC_IOFPGA_VAL=0x01505327
MACSEC_36P_LC_IOFPGA_VAL=0x01505328
COH_MACSEC_6P_LC_IOFPGA_VAL=0x0011986C
FRETTA_RP_PRECISA_IOFPGA_VAL=0x01573247
FRETTA_RP_PRECISA2_IOFPGA_VAL=0x15813136
POTENZA_PLUS_100_LC_IOFPGA_VAL=0x01505329
POTENZA_PLUS_MPA_LC_IOFPGA_VAL=0x0150532C
POTENZA_PLUS_MPA_LC_TCAM_IOFPGA_VAL=0x0150532D #Check this
POTENZA_PLUS_10C_A_LC_IOFPGA_VAL=0x01601004   #POTENZA+10C-A and POTENZA+10C-AT
VIGOR_400_LC_IOFPGA_VAL=0x01601000
VIGOR_400_LC_TCAM_IOFPGA_VAL=0x01601001
VIGOR_100_LC_IOFPGA_VAL=0x01601005
VIGOR_100_LC_TCAM_IOFPGA_VAL=0x01601006
VIGOR_DCO_LC_TCAM_IOFPGA_VAL=0x01601007
VIGOR_MOD_LC_IOFPGA_VAL=0x01601008
MIFPGA_ID_PEYTO=0xb5171fd3
MIFPGA_ID_PEYTO_HD=0xd5171fd3
MIFPGA_ID_PEYTO_SE=0xa5171fd3
MIFPGA_ID_MOD_LC=0x01518600
MIFPGA_ID_MOD_LC_SE=0x01518601
MIFPGA_ID_PEYTO_CC=0xf5171fd3
BIFROST_T_RP_IOFPGA_VAL=0xC0171FD3
TURIN_CR_RP_IOFPGA_VAL=0xE0171FD3
#/* TBD: Change to right value */
TURIN_CR_MACSEC_RP_IOFPGA_VAL=0xE0763B90

FRETTA_RP_EYRIE_IOFPGA_VAL=0x00986FEB
FRETTA_EYRIE_LC_SE_IOFPGA_VAL=0x80744254
FRETTA_EYRIE_LC_NON_SE_IOFPGA_VAL=0x90744254
MIFPGA_ID_FRETTA_EYRIE_LC_NON_SE=0xa11ecf03
MIFPGA_ID_FRETTA_EYRIE_LC_SE=0xb11ecf03
MIFPGA_ID_VIGOR_MOD_LC=0x01518605

# description: This function will do the following
# 1. Locate the MIFGPA device e0900000
# 2. Get the BAR0 base address for MIFPGA 
# 3. Get the BAR3(MPA CTRL) for MIFPGA 
# 4. If the platform is Peyto or fixed MPA based ones, it 
#    will init bar3. If this is MOD LC then it will take  
#    BAR2 
function init_mifpga_base_add () {

    PCI_BASE_ADDR_BAR0_MIFPGA=0x`lspci -vd $MIFPGA_VENDOR:$MIFPGA_ID | grep "Memory at" | cut -d' ' -f3 | head -n1 | cut -d " " -f1`

    reg_fpga_id=0x00000004
    base_add=$PCI_BASE_ADDR_BAR0_MIFPGA
    reg_addr=$(printf 0x%08X $(($base_add + $reg_fpga_id)))
    val=0x`pcimemread $reg_addr 4 | grep ":" | awk '{print $3}'`

    #based on FPGA ID assign value of bar_num for MPA CTRL 
    # EYRIE_TBD 
    case $val in 
         $MIFPGA_ID_PEYTO | $MIFPGA_ID_PEYTO_CC | $MIFPGA_ID_PEYTO_HD | $MIFPGA_ID_PEYTO_SE)
                         #Peyto platform 
                         bar_num=4
                         ;;
         $MIFPGA_ID_MOD_LC | $MIFPGA_ID_MOD_LC_SE) 
                         # MOD LC card 
                         bar_num=4
                         ;;
         $MIFPGA_ID_FRETTA_EYRIE_LC_NON_SE | $MIFPGA_ID_FRETTA_EYRIE_LC_SE)
                         # EYRIE LC card
                         bar_num=4
                         ;;
         $MIFPGA_ID_VIGOR_MOD_LC)
                         # VIGOR MOD LC card
                         bar_num=3
                         ;;
         *)echo "$val platform not found "
           exit 1
    esac
         
    PCI_BASE_ADDR_BAR_MPA_CTRL_MIFPGA=0x`lspci -vd $MIFPGA_VENDOR:$MIFPGA_ID | grep "Memory at" | cut -d' ' -f3`
    MPA_BAR_ADD=$(echo $PCI_BASE_ADDR_BAR_MPA_CTRL_MIFPGA | cut -d " " -f $bar_num)
}

function init_iofpga_base_addr_zermatt () {
    PCI_BASE_ADDR_CPU_IOFPGA=0x`lspci -vd $IOFPGA_VENDOR:$CPU_BOARD_IOFPGA_ID | grep "Memory at" | cut -d' ' -f3 | head -n1`
    # The Base IOFPGA has 2 memory bars on turin, so there are 2 "Memory at"
    # lines in the output.  We need the address of bar 0, which has a size
    # of 256M or 128M.
    PCI_BASE_ADDR_BASE_IOFPGA=0x`lspci -vd $IOFPGA_VENDOR:$BASE_BOARD_IOFPGA_ID | grep "Memory at" | egrep "256M|128M" | cut -d' ' -f3`
    PCI_BASE_ADDR_MEZZ_IOFPGA=0x`lspci -vd $IOFPGA_VENDOR:$MEZZ_BOARD_IOFPGA_ID | grep "Memory at" | cut -d' ' -f3`
}

#
# Function for finding IOFPGA base address with lspci tool and set the
# IOFPGA_BASE_ADDR variable if is not yet set
# RP_IOFPGA_ID should use CPU_BOARD_IOFPGA_ID 
# OTHER_IOFPGA_ID should use BASE_BOARD_IOFPGA_ID  
#

RP_IOFPGA_ID=100
OTHER_IOFPGA_ID=101

function init_iofpga_base_addr_fretta () {
    local pci_addr=0
    local iofpga_id=$OTHER_IOFPGA_ID
    get_board_type_fretta

    if [ "$BOARDTYPE" == "RP" ]; then
        iofpga_id=$RP_IOFPGA_ID
    fi
    
    pci_addr=`lspci -vd $IOFPGA_VENDOR:$iofpga_id | grep "Memory at" | cut -d' ' -f3`
    IOFPGA_BASE_ADDR=$((0x$(echo $pci_addr | cut -d " " -f 1) + 0x8000000))
    IOFPGA_BAR1_ADDR=0x$(echo $pci_addr | cut -d " " -f 2)
}

function init_dbfpga_base_addr_fretta () {
    local pci_addr=0
    local iofpga_id=104

    pci_addr=`lspci -vd $IOFPGA_VENDOR:$iofpga_id | grep "Memory at" | cut -d' ' -f3 | head -n1`
    DBFPGA_BASE_ADDR=$((0x$(echo $pci_addr | cut -d " " -f 1) + 0x8000000))
}
function platform_type_get {
    PLATFORM="$(cat /proc/cmdline|awk -F"platform=" '{print $2}'|cut -d' ' -f1)"
}

# Note that output of pcimemread show a messy output to parse:
#
# PCI memory 0xe8000000 align memory 0xe8000000 (mmap addr 0x7ff1b2a35000(0x2000))
#        e8000000 : 00000022  "...
# So we have to use a combination of tr and cut operation to get the job done
# Arg $1
# 0 = Cpu  Board IOFPGA 
# 1 = Base Board IOFPGA 
# 2 = Mezz Board IOFPGA 
#
# Arg $2: Register address 
function iofpga_reg_read_zermatt () {
    init_iofpga_base_addr_zermatt 
    local iofpga_base_addr=0
    local reg_addr=0
    local val=0
    
    # Check instance number  
    case "$1" in
         0) 
           # Store  CPU IOFPGA 
           iofpga_base_addr=$PCI_BASE_ADDR_CPU_IOFPGA
           ;;
         1)
           # Store BASE IOFPGA
           iofpga_base_addr=$PCI_BASE_ADDR_BASE_IOFPGA 
         ;;
         2)
           # Store MEZZ IOFPGA 
           iofpga_base_addr=$PCI_BASE_ADDR_MEZZ_IOFPGA
         ;;
         *)echo "Instance $1 is not valid, choose 0 , 1 or 2"
           exit 1
    esac

    # Read the value 
    reg_addr=$(printf 0x%08X $(($iofpga_base_addr + $2 + $IOFPGA_OFFSET)))
    val=0x`pcimemread $reg_addr 4 | grep ":" | awk '{print $3}'`
    echo $(printf 0x%08X $val)
}

#
# Note that output of pcimemread show a messy output to parse:
#
# PCI memory 0xe8000000 align memory 0xe8000000 (mmap addr 0x7ff1b2a35000(0x2000))
#        e8000000 : 00000022  "...
# So we have to use a combination of tr and cut operation to get the job done
function iofpga_reg_read_fretta () {
    init_iofpga_base_addr_fretta
    local reg_addr=$(printf 0x%08X $(($IOFPGA_BASE_ADDR + $1)))
    local val=0x`pcimemread $reg_addr 4 | grep ":" | awk '{print $3}'`
    echo $(printf 0x%08X $val)
}

# arguments: <bar> <addr>
# only to read CPU IOFPGA [Device ID : 0x101]
#  - Valid Bar's - 0 [default] / 1
function iofpga_reg_read_fretta_v2 () {
    if [ "$#" -ne 2 ]; then
        echo "Illegal number of parameters"
        return
    fi

    init_iofpga_base_addr_fretta
    local bar=$1
    local base_addr
    if [ "$bar" -eq 1 ]; then
        base_addr=$IOFPGA_BAR1_ADDR
    else
        base_addr=$IOFPGA_BASE_ADDR
    fi
    local reg_addr=$(printf 0x%08X $(($base_addr + $2)))
    local val=0x`pcimemread $reg_addr 4 | grep ":" | awk '{print $3}'`
    echo $(printf 0x%08X $val)
}

#
# PCI read from DB fpga (0x104) from a specific offset $1
#
function iofpga_db_reg_read_fretta () {
    init_dbfpga_base_addr_fretta
    local reg_addr=$(printf 0x%08X $(($DBFPGA_BASE_ADDR + $1)))
    local val=0x`pcimemread $reg_addr 4 | grep ":" | awk '{print $3}'`
    echo $(printf 0x%08X $val)
}

# NOTE : This is always a 4 byte write
# Functions for accessing IOFPGA registers
# We break it down to which IOFGPA instance to access
#
# Arg $1
# 0 = Cpu  Board IOFPGA
# 1 = Base Board IOFPGA
# 2 = Mezz Board IOFPGA
#
# Arg $2: Register address
# Arg $3: Value to write
function iofpga_reg_write_zermatt () {
    init_iofpga_base_addr_zermatt
    local iofpga_base_addr=0
    local reg_addr=0
    local val=0

    # Check instance number
    case "$1" in
         0)
           # Store CPU IOFPGA
           iofpga_base_addr=$PCI_BASE_ADDR_CPU_IOFPGA
           ;;
         1)
           # Store BASE IOFPGA
           iofpga_base_addr=$PCI_BASE_ADDR_BASE_IOFPGA
           ;;
         2)
           # Store MEZZ IOFPGA
           iofpga_base_addr=$PCI_BASE_ADDR_MEZZ_IOFPGA
           ;;
         *)echo "Instance $1 is not valid, choose 0 , 1 or 2"
           exit 1
    esac

    # Write the value
    reg_addr=$(printf 0x%08X $(($iofpga_base_addr + $2 + $IOFPGA_OFFSET)))
    local val="$3"
    /usr/bin/logger -t pcimemwrite "$(pcimemwrite $reg_addr 4 $val)"
}

#
# Functions for accessing IOFPGA registers
#

#
# arguments: <bar> <addr> <value>
function iofpga_reg_write_fretta_internal() {

    local base_addr
    
    init_iofpga_base_addr_fretta
    local bar=$1
    
    if [ "$bar" -eq 0 ]; then
        base_addr=$IOFPGA_BASE_ADDR
    else
        base_addr=$IOFPGA_BAR1_ADDR
    fi
        
    local reg_addr=$(printf 0x%08X $(($base_addr + $2)) )
    local val=$3

    /usr/bin/logger -t pcimemwrite "$(pcimemwrite $reg_addr 4 $val)"
}

function iofpga_reg_write_fretta () {
    # By default write to Bar 0
    iofpga_reg_write_fretta_internal 0 $1 $2
}

function iofpga_reg_write_fretta_bar1 () {
    iofpga_reg_write_fretta_internal 1 $1 $2
}

#
# arguments: <bar> <addr> <value>
function iofpga_db_reg_write_fretta_internal() {
    local base_addr
    init_dbfpga_base_addr_fretta
    local bar=$1

    if [ "$bar" -eq 0 ]; then
        base_addr=$DBFPGA_BASE_ADDR
    fi

    local reg_addr=$(printf 0x%08X $(($base_addr + $2)) )
    local val=$3

    /usr/bin/logger -t pcimemwrite "$(pcimemwrite $reg_addr 4 $val)"
}

function iofpga_db_reg_write_fretta () {
    # By default write to Bar 0
    iofpga_db_reg_write_fretta_internal 0 $1 $2
}
#
# Description: This fuction will do a read of BAR0 and MPA CTRL bar 
# Input Arguments for this function 
# 
# $1 = Bar number 0 and 3 are allowed 
# $2 = Register address to read 
# 
# Output:
# register read value
function mifpga_reg_read () {
    init_mifpga_base_add

    case "$1" in
        0)
          mifpga_base_add=$PCI_BASE_ADDR_BAR0_MIFPGA
          ;;
        3)
          mifpga_base_add=0x$MPA_BAR_ADD
          ;;
        *)echo "Unknow $1 bar used , choose 0 or 3"
          exit 1

     esac

     reg_add_read=$(printf 0x%08X $(($mifpga_base_add + $2)))
     value_read=0x`pcimemread $reg_add_read 4 | grep ":" | awk '{print $3}'`
     echo $(printf 0x%08x $value_read)
}

#
# Description: This fuction will do a Write of BAR0 and MPA CTRL bar 
# Input Arguments for this function 
# 
# $1 = Bar number 0 and 3 are allowed 
# $2 = Register address to read 
# $3 = Value to write
#
# Output:
# None 
function mifpga_reg_write () {
    init_mifpga_base_add

    case "$1" in
        0)
          #echo "BARO selected"
          mifpga_base_add=$PCI_BASE_ADDR_BAR0_MIFPGA
          ;;
        3)
          #echo "MPA CTRL BAR selected "
          mifpga_base_add=0x$MPA_BAR_ADD
          ;;
        *)echo "Unknow $1 bar used , choose 0 or 3"
          exit 1

     esac

     write_add=$(printf 0x%08X $(($mifpga_base_add + $2)))
     /usr/bin/logger -t pcimemwrite "$(pcimemwrite $write_add 4 $3)"
}

#
# Log to file and console
#
function platform_log_console_ext
{
    declare -F platform_log_console >& /dev/null
    
    if [ $? -eq 0 ]; then
        platform_log_console "$@"
    else
        if [ "${PARTMNT}" ]; then 
            echo "$@" >>  "${PARTMNT}/root/log.txt"
        else   
            echo "$@" 
        fi    
    fi
}


# Board type value set in BIOS
# 0 - SC
# 1 - RP
# 2 - LC
# etc
function get_board_type_fretta {
  ARCH=`arch`
  if [ "$ARCH" == "armv7l" ]; then
    BOARDTYPE=$(get_cctrl_board_type)
  else 
    BOARDTYPE="$(cat /proc/cmdline|awk -F"boardtype=" '{print $2}'|cut -d' ' -f1)"
    if [ -z ${BOARDTYPE} ];then
      BOARDTYPE="RP"
    fi
  fi
}

function get_board_type_zermatt {
    BOARDTYPE=`cat /proc/cmdline | grep boardtype=`
    if [ -n "$BOARDTYPE" ]; then
	BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    else 
	BOARDTYPE="RP"
    fi 
}

function pd_get_card_inst_from_card_type_fretta () {
    local lcard_value_f
    local lcard_inst_f

    init_iofpga_base_addr_fretta

    lcard_value_f=$(iofpga_reg_read_fretta 0x4)
    platform_log_console_ext "lcard_value_f is ${lcard_value_f}"
   
    case ${1} in
        "RP")
            case ${lcard_value_f} in

                "${FRETTA_RP_PRECISA_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-RP,FRETTA-SYNC-RP"
                ;;
                "${FRETTA_RP_PRECISA2_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-RP,FRETTA-SYNC-RP2"
                ;;
                "${FRETTA_RP_EYRIE_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-HYBRID-RP,FRETTA-NON-SYNC-RP"
                ;;
                "${FRETTA_RP_SUPA_IOFPGA_VAL}")                         ;&
                "${FRETTA_RP_SUPB_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-RP,FRETTA-NON-SYNC-RP"
                ;;
                *)  lcard_inst_f="FRETTA-RP,FRETTA-NON-SYNC-RP"
                ;;
            esac
            ;;
        "LC")
            case "$lcard_value_f" in
                # LC POTENZA 100 Coherent
                # 6.1.x releases do not expand card_instance_groups to 
                # card_instances at build time.
                # SEC is added for backword caompatibility.
                "${COH_MACSEC_6P_LC_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-NON-SYNC-LC,FRETTA-SEC-LC,SEC,COH-LC"
                ;;
                # LC POTENZA MACSEC
                # 6.1.x releases do not expand card_instance_groups to 
                # card_instances at build time.
                # SEC is added for backword caompatibility.
                "${MACSEC_18P_LC_IOFPGA_VAL}"|"${MACSEC_36P_LC_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-NON-SYNC-LC,FRETTA-SEC-LC,SEC,OPTICS-LC"
                ;;
                # LC POTENZA+ 10C-A and LC POTENZA+ 10C-AT
                "${POTENZA_PLUS_10C_A_LC_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,FRETTA-SEC-LC,SEC,OPTICS-LC,POTENZA-P10C-LC"
                ;;
                "${FRETTA_EYRIE_LC_NON_SE_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-HYBRID-LC,FRETTA-SYNC-LC,FRETTA-SEC-LC,SEC,OPTICS-LC"
                ;;
                "${FRETTA_EYRIE_LC_SE_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-HYBRID-LC,FRETTA-SYNC-LC,FRETTA-SEC-LC,SEC,OPTICS-LC"
                ;;
                # LC POTENZA+ 100
                "${POTENZA_PLUS_100_LC_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,OPTICS-LC"
                ;;
                # LC POT+ MPA
                "${POTENZA_PLUS_MPA_LC_IOFPGA_VAL}")    ;&
                "${POTENZA_PLUS_MPA_LC_TCAM_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,FRETTA-SEC-LC,SEC,OPTICS-LC,COH-LC"
                ;;
                # LC VIGOR 400 
                "${VIGOR_400_LC_IOFPGA_VAL}")   ;&
                "${VIGOR_400_LC_TCAM_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,OPTICS-LC,VIGOR-LC"
                ;;
                # LC VIGOR 100 SE
                # LC VIGOR DCO
                # Vigor VE cards with One J2 device
                "${VIGOR_100_LC_TCAM_IOFPGA_VAL}") ;&
                "${VIGOR_DCO_LC_TCAM_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,OPTICS-LC,VIGOR-VE"
                ;;
                # LC VIGOR 100 Base
                "${VIGOR_100_LC_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,FRETTA-SEC-LC,OPTICS-LC,VIGOR-VE,VIGOR-100-BASE-LC"
                ;;
                # LC VIGOR MOD
                "${VIGOR_MOD_LC_IOFPGA_VAL}")
                lcard_inst_f="FRETTA-LC,FRETTA-SYNC-LC,FRETTA-SEC-LC,OPTICS-LC,VIGOR-VE,VIGOR-MOD-LC"
                ;;
 
                *) lcard_inst_f="FRETTA-LC,FRETTA-NON-SYNC-LC,OPTICS-LC"
                ;;
            esac 
            ;; 
        "XC") lcard_inst_f="FRETTA-XC" ;;
        "FC") lcard_inst_f="FRETTA-FC" ;;
        *)    lcard_inst_f="FRETTA-RP" ;;
      esac

    eval "$2=$lcard_inst_f"
    platform_log_console_ext "FRETTA: card_inst=${lcard_inst_f} card_type=${1}"
}

function pd_get_card_inst_from_card_type_zermatt () {
    local lcard_value_z
    local lcard_inst_z

    init_iofpga_base_addr_zermatt

    lcard_value_z=$(iofpga_reg_read_zermatt 0 0x4)
    platform_log_console_ext "card_value is ${lcard_value_z}"
   
    case ${1} in
        "RP")
             case ${lcard_value_z} in
                "${ZERMATT_CR_IOFPGA_VAL}")           ;&
                "${ZERMATT_RP_IOFPGA_VAL}")
                    lcard_inst_z="ZERMATT-RP"
                    ;;
                "${TURIN_RP_IOFPGA_VAL}")
                    lcard_inst_z="TURIN-RP,TURIN-SYNC-RP"
                    ;;
                "${TAIHU_RP_IOFPGA_VAL}")
                    lcard_inst_z="TURIN-RP,TURIN-NON-SYNC-RP"
                    ;;
                "${WINTERFELL_RP_IOFPGA_VAL}")
                    lcard_inst_z="WINTERFELL-RP"
                    ;;
                "${OLDCASTLE_RP_IOFPGA_VAL}")         ;&
                "${OLDCASTLE_CR_IOFPGA_VAL}")
                    lcard_inst_z="OLDCASTLE-RP"
                    ;;
                "${PEYTO_RP_IOFPGA_VAL}")         ;&
                "${PEYTO_CC_RP_IOFPGA_VAL}")      ;&
                "${PEYTO_CR_IOFPGA_VAL}")         ;&
                "${PEYTO_CR_HD_IOFPGA_VAL}")
                    lcard_inst_z="PEYTO-RP"
                    ;;
                "${TORTIN_RP_IOFPGA_VAL}")        ;&
                "${TORTIN_RP_16G_IOFPGA_VAL}")     
                    lcard_inst_z="TORTIN-RP"
                    ;;
                "${PYKE_RP_IOFPGA_VAL}")
                    lcard_inst_z="PYKE-RP"
                    ;;
                "${BIFROST_T_RP_IOFPGA_VAL}")
                    lcard_inst_z="BIFROSTT-RP"
                    ;;
                "${TURIN_CR_RP_IOFPGA_VAL}")      ;&
                "${TURIN_CR_MACSEC_RP_IOFPGA_VAL}")
                    lcard_inst_z="TURINCR-RP"
                    ;;
                *)  lcard_inst_z="ZERMATT-RP"
                    ;;
             esac
            ;;
        "LC")
            case "$lcard_value_z" in
                "${ZERMATT_CR_IOFPGA_VAL}")           ;&
                "${ZERMATT_RP_IOFPGA_VAL}")
                   lcard_inst_z="ZERMATT-LC"
                   ;;
                "${TURIN_RP_IOFPGA_VAL}")
                   lcard_inst_z="TURIN-LC,TURIN-SYNC-LC"
                   ;;
                "${TAIHU_RP_IOFPGA_VAL}")
                   lcard_inst_z="TURIN-LC,TURIN-NON-SYNC-LC"
                   ;;
                "${WINTERFELL_RP_IOFPGA_VAL}")
                   lcard_inst_z="WINTERFELL-LC"
                   ;;
                "${OLDCASTLE_RP_IOFPGA_VAL}")         ;&
                "${OLDCASTLE_CR_IOFPGA_VAL}")
                   lcard_inst_z="OLDCASTLE-LC"
                   ;;
                "${PEYTO_RP_IOFPGA_VAL}")         ;&
                "${PEYTO_CC_RP_IOFPGA_VAL}")      ;&
                "${PEYTO_CR_IOFPGA_VAL}")         ;&
                "${PEYTO_CR_HD_IOFPGA_VAL}")
                   lcard_inst_z="PEYTO-LC"
                   ;;
                "${TORTIN_RP_IOFPGA_VAL}")       ;&
                "${TORTIN_RP_16G_IOFPGA_VAL}")     
                   lcard_inst_z="TORTIN-LC"
                   ;;
                "${PYKE_RP_IOFPGA_VAL}")
                   lcard_inst_z="PYKE-LC"
                   ;;
                "${BIFROST_T_RP_IOFPGA_VAL}")
                   lcard_inst_z="BIFROSTT-LC"
                   ;;
                "${TURIN_CR_RP_IOFPGA_VAL}")      ;&
                "${TURIN_CR_MACSEC_RP_IOFPGA_VAL}")
                    lcard_inst_z="TURINCR-LC"
                    ;;
                *)  lcard_inst_z="ZERMATT-LC"
                   ;;
            esac 
            ;; 
        *)    lcard_inst_z="ZERMATT-RP" 
            ;;
      esac

    eval "$2=$lcard_inst_z"
    platform_log_console_ext "ZERMATT: card_inst=${lcard_inst_z} card_type=${1}"
}

#
# Usage : pd_get_card_inst_from_card_type card_type card_inst
function pd_get_card_inst_from_card_type () {
    local lcard_inst 

    platform_type_get

    if [ "${PLATFORM}" == "fretta" ]; then
        pd_get_card_inst_from_card_type_fretta $1 lcard_inst
    elif [ "${PLATFORM}" == "zermatt" ]; then
        pd_get_card_inst_from_card_type_zermatt $1 lcard_inst
    elif [ "${PLATFORM}" == "iosxrwbd" ]; then
        IS_MC=$(/usr/bin/xr_sysctl -n cisco.ios.xr.multichassis)

        case ${1} in
        "RP")
            if [[ -z ${IS_MC} ]]; then 
                lcard_inst="TURIN-RP,IOSXRWBD-SLC-RP"
            else
                lcard_inst="TURIN-RP,IOSXRWBD-DDC-RP"
            fi
            ;;
        "CC")
            lcard_inst="IOSXRWBD-CC"
            ;;
        "SW")
            lcard_inst="IOSXRWBD-SW"
            ;;
        "FC")
            lcard_inst="IOSXRWBD-FC"
            ;;
        "LC")
            # For sysadmin/host VM type , set as IOSXRWBD-LC
            if [ ${4} == "2" -o ${4} == "1" ]; then
                lcard_inst="IOSXRWBD-LC"
            else
                if [[ -z ${IS_MC} ]]; then
                    lcard_inst="TURIN-LC,IOSXRWBD-SLC-LC"
                else 
                    lcard_inst="TURIN-LC,IOSXRWBD-DDC-LC"
                fi
            fi

            ;;
         *)  lcard_inst_z="TURIN-LC"
            ;;
        esac
    else
        platform_log_console_ext "pd_get_card_inst_from_card_type() wrong platform ${PLATFORM}"
    fi

    platform_log_console_ext "pd_get_card_inst_from_card_type() card_inst=${lcard_inst}"

    eval "$2=${lcard_inst}"
}


