#!/bin/bash
# Author : Mahesh Singh
# Copyright (c) 2018 by Cisco Systems, Inc.
# All rights reserved.

###############################################################
## source the required scripts
###############################################################

if [ -z "${PD_DYNAMIC}" ];then
    PD_DYNAMIC="etc/init.d/spirit_pd.sh"
fi
if [ -f "${PD_DYNAMIC}" ]; then
    step_log_console "Sourcing "${PD_DYNAMIC}" script"
    . ${PD_DYNAMIC}     
fi

###################Global Variables ###########################
readonly PKG_VER=1.0

###############################################################
##              functions 
###############################################################
function step_log_console {
    local log_msg=${1}
    local time_str=$(date)
    echo "${time_str}: ${log_msg}" >&105
    # Also log console message to file
    echo "${time_str}: ${log_msg}" >&107
    if [ $USE_FINAL -gt 0 ]; then
        sync
    fi
}
readonly -f step_log_console

function get_packages_to_be_installed {
    local iso_type=${1}
    local -a pkgs=("${!2}")

    unset packages_to_be_installed
    if [ ${iso_type} == ${PKG_VER} ]; then
        for file in ${pkgs[@]}
        do
            packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
        done
    else
    for file in ${pkgs[@]}
    do
        is_rp_package=0
        is_lc_package=0
        is_sc_package=0
        is_xc_package=0
        is_fc_package=0
        is_common_package=0
        if [[ ${file} == *.all-* ]] ; then
            is_common_package=1;
        else
            if [[ ${file} == *.rp[-,_]* ]] ; then
                is_rp_package=1;
            fi
            if [[ ${file} == *[.,_]sc* ]] ; then
                is_sc_package=1;
            fi
            if [[ ${file} == *[.,_]lc* ]] ; then
                is_lc_package=1;
            fi
            if [[ ${file} == *[.,_]xc* ]] ; then
                is_xc_package=1;
            fi
            if [[ ${file} == *[.,_]fc* ]] ; then
                is_fc_package=1;
            fi
        fi
        if [ "$is_rp_package" -ne "1" -a                                      \
             "$is_sc_package" -ne "1" -a                                      \
             "$is_lc_package" -ne "1" -a                                      \
             "$is_xc_package" -ne "1" -a                                      \
             "$is_fc_package" -ne "1" -a                                      \
             "$is_common_package" -ne "1" ]; then
            step_log_console "Error! Pkg ${file} is not applicable to any card"
        fi
        case ${BOARDTYPE} in
            LC)
               if [ "$is_lc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            SC)
               if [ "$is_sc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            XC)
               if [ "$is_xc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            FC)
               if [ "$is_fc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            *)
              # default is RP 
               if [ "$is_rp_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
        esac
    done
    fi
}

function install_initrd {
    root_dir=$1
    nbi_dir=$2
    input_vm_type=$3
    local -a packages

    cd ${root_dir}
    time zcat -f ${nbi_dir}/sysadmin-nbi-initrd.img | cpio -id 
    
    # Get card instance information and add it in the mount
    declare -f pd_get_card_inst > /dev/null
    if [ "$?" -eq 0 ]; then
        local CARD_INST
        pd_get_card_inst CARD_INST
        echo "$BOARDTYPE $CARD_INST" > $CARD_INST_FILE
        step_log_console $"Using board ${BOARDTYPE} and inst ${CARD_INST}"
    fi

    step_log_console "sysadmin-vm: Starting $BOARDTYPE $root_dir based RPM installation"

    # Obtain the list of packages to be installed
    if [ ${input_vm_type} == "host" ]; then
        packages=(rpm/*hostos*)
    else
        packages=(rpm/*)
    fi

    get_packages_to_be_installed "${PKG_VER}" packages[@]

    for file in ${packages_to_be_installed[@]}
    do
        if [ ${PKG_VER} == "1.0" ]; then
            if [ ${file} == *thirdparty-libexec* ]; then
                step_log_console $"Ignoring Install RPM ${file} of size "
            else
                SUPP_CARDS=()
                excl_card=()
                #SUPPCARDS holds a list of supported cards by this rpm; comma separated.
                supp_card=`chroot ${root_dir} rpm -qp --qf '%{SUPPCARDS}\n' ${file}`
                SUPP_CARDS=(${supp_card//,/ })
                board=`echo ${BOARDTYPE} | tr "a-z" "A-Z"`
                for card in ${SUPP_CARDS[@]}
                do
                    card=`echo $card | tr "a-z" "A-Z"`
                    if [ $card != ${BOARDTYPE} -a  $card != ALL ]; then
                        excl_card[${#excl_card[*]}]=`echo $card | tr "A-Z" "a-z"`
                    fi
                done
                declare -a excl
                excl=()
                prefix=`chroot ${root_dir} rpm -qp --qf '%{PREFIXES}\n' ${file}`
                pkg_name=`chroot ${root_dir} rpm -qp --qf '%{NAME}\n' ${file}`
                version=`chroot ${root_dir} rpm -qp --qf '%{VERSION}\n' ${file}`
                release=`chroot ${root_dir} rpm -qp --qf '%{RELEASE}\n' ${file}`
                rpm_vmtype=`chroot ${root_dir} rpm -qp --qf '%{VMTYPE}\n' ${file}`
                
                if [ "$rpm_vmtype" ==  $input_vm_type ]; then
                    excl_path=${prefix}/${pkg_name}-${version}-${release}
                    for card in ${excl_card[@]}
                    do
                        excl[${#excl[*]}]=--excludepath=${excl_path}/${card}
                    done
                    exclude=`echo ${excl[@]}`
                    step_log_console $"Install RPM with ${exclude}"
                    STARTRPM=$(date +%s)
                    chroot ${root_dir} rpm -i ${exclude} --nodeps ${file} >> /tmp/rpmlog 2>&1
                    ENDRPM=$(date +%s)
                    DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                    SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                    step_log_console $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
                else
                    step_log_console $"Skipping ${file}"
                    step_log_console $"Skipping ${file}"
                fi
            fi
       else
           case ${file} in
               *.rpm) 
                 if [ ${file} == *thirdparty-libexec* -o \
                      ${file} == *hostos* ] ; then
                    step_log_console $"Ignoring Install RPM ${file} of size "
                 else
                 STARTRPM=$(date +%s)
                 chroot ${root_dir} rpm -i --nodeps ${file} >> /tmp/rpmlog 2>&1
                 ENDRPM=$(date +%s)
                 DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                 SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                 step_log_console $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
                 fi
                 ;;
           esac
        fi
    done
}

function install_smus {
    root_dir=$1
    input_vm_type=$2
    SMU_INST_SCRIPT_PATH=/usr/bin/calv_memboot_smu_install.py 
    host_smu_install_options="-host -chroot -nodel"
    Calvados_smu_install_options=""
    options=`eval 'echo ${input_vm_type}_smu_install_options'`
    if [ -f ${SMU_INST_SCRIPT_PATH} ]; then
         chmod 744 ${SMU_INST_SCRIPT_PATH}
         step_log_console "Install calvados SMUs"
         STARTRPM=$(date +%s)
         ${SMU_INST_SCRIPT_PATH} -b "${BOARDTYPE}" -c "${root_dir}" ${!options} |tee /tmp/calv_memboot_smu_install.log 2>&1
         ENDRPM=$(date +%s)
         DIFFRPM=$(( $ENDRPM - $STARTRPM ))
         step_log_console "SMU installation took ${DIFFRPM} sec"
    fi
}

function install_setup_symlink_and_dirs {
    root_dir=$1
    if [ -f ${root_dir}/usr/bin/calv_setup_ldpath.py ]; then
         chmod 744 ${root_dir}/usr/bin/calv_setup_ldpath.py
         step_log_console "Install sysadmin symlink setup"
         STARTRPM=$(date +%s)
         arg="`echo ${excl_card[@]}`"
         ${root_dir}/usr/bin/calv_setup_ldpath.py ${root_dir} "${arg}" > /tmp/calv_setup_ldpath.log 2>&1
         ENDRPM=$(date +%s)
         DIFFRPM=$(( $ENDRPM - $STARTRPM ))
         step_log_console "sysadmin-vm symlink setup took ${DIFFRPM} sec"
    fi

    step_log_console "Install sysadmin dir setup"
}

function install_setup_dirs {
    root_dir=$1
    vm_type=$2
    LV_ID=$3
    mkdir -p ${root_dir}/proc ${root_dir}/sys ${root_dir}/root/.ssh
    cp ${SSHKEY_DIR}/${vm_type}/* ${root_dir}/root/.ssh/
    mkdir -p ${root_dir}/dev 
    chmod -R 700 ${root_dir}/root
    cd / > /dev/null 2>&1
    mount_efi_and_proc ${root_dir}                                        
    mount_pd_fs ${root_dir}
    if [ "${vm_type}" == "calvados" ]; then
        vm_type="sysadmin-vm"
    fi 
    chroot ${root_dir} /etc/init.d/calvados_patch_lxc_iso.sh "/"  ${LVG_NAME} \
                 ${LV_ID} ${vm_type} 
    #unmount /proc and other FS passed to ${PARTMNT}
    umount_efi_and_proc ${root_dir}
    umount_pd_fs ${root_dir}
    run_depmod ${root_dir}
    sync
    sync
    SIZE=$(du -hs --exclude="/proc" --exclude="/sys" ${root_dir} | awk '{print $1}')
}

function install_calvados {
    dev_name=${1}
    nbi_dir=${2}
    part_mnt=$(mktemp -d /tmp/partprep.XXXXXX)
    mount -o offset=${3} ${dev_name} ${part_mnt}
    #second parameter should be vmtype parameter from rpms
    install_initrd ${part_mnt} ${nbi_dir} "Calvados"
    install_smus ${part_mnt}  "Calvados"
    install_setup_symlink_and_dirs ${part_mnt}
    install_setup_dirs ${part_mnt} "calvados" 0
    find ${part_mnt} | sort -u > ${part_mnt}/files.txt
    umount ${part_mnt} && rmdir ${part_mnt}
}

function install_host {
    dev_name=$1
    nbi_dir=$2
    part_mnt=$(mktemp -d /tmp/partprep.XXXXXX)
    mount -o discard ${dev_name} ${part_mnt}
    install_initrd ${part_mnt} ${nbi_dir} "host"
    cp $nbi_dir/fm.nbi $part_mnt/boot/sflc-auto.img
    chroot ${part_mnt} mkdir -p /etc/libvirt/hooks/
    chroot ${part_mnt} ln -s /etc/init.d/lxc_hook.sh /etc/libvirt/hooks/lxc
    install_smus ${part_mnt} "host"
    install_setup_dirs ${part_mnt} "hostos" 0
    find ${part_mnt} | sort -u > ${part_mnt}/files.txt
    umount ${part_mnt} && rmdir ${part_mnt}
}

function run_depmod {
    for d in $1/lib/modules/*;
    do

        if [ -d $d ]; then
            depmod -a -b $1 `basename $d`
        fi
    done
}
